import grpc
import logging

from concurrent import futures
from perxis.collections import collections_pb2_grpc
from perxis.environments import environments_pb2_grpc
from perxis.roles import roles_pb2_grpc
from perxis.clients import clients_pb2_grpc
from perxis.extensions import extension_pb2_grpc, manager_pb2_grpc, manager_pb2
from perxis.interceptors import header_adder_interceptor

from servicer import Servicer
from constants.extension import get_extension_descriptor

logging.basicConfig(level=logging.INFO)

logger = logging.getLogger(__name__)


def main():
    my_extension_descriptor = get_extension_descriptor(host="demo-ext-backend:50051")

    logger.info(f"Инициализация сервиса расширения {my_extension_descriptor.extension}")

    interceptor = header_adder_interceptor(
        'x-perxis-access', 'system'
    )
    with grpc.insecure_channel("extension-manager:9030") as extensions_manager_channel:
        intercept_channel_extensions_manager_channel = grpc.intercept_channel(extensions_manager_channel, interceptor)

        ext_manager_stub = manager_pb2_grpc.ExtensionManagerStub(intercept_channel_extensions_manager_channel)

        registered_extensions: manager_pb2.ListExtensionsResponse = ext_manager_stub.ListExtensions(manager_pb2.ListExtensionsRequest())

        for ext in registered_extensions.extensions:
            if ext.extension == my_extension_descriptor.extension:
                if ext.version != my_extension_descriptor.version:
                    ext_manager_stub.UnregisterExtensions(
                        manager_pb2.UnregisterExtensionsRequest(
                            extensions=[my_extension_descriptor]
                        )
                    )

                    logger.info(
                        f"Изменилась версия расширения {my_extension_descriptor.extension} ({ext.version} -> {my_extension_descriptor.version})"
                    )

        ext_manager_stub.RegisterExtensions(manager_pb2.RegisterExtensionsRequest(
            extensions=[my_extension_descriptor]
        ))

        logger.info(f"Расширение {my_extension_descriptor.extension} зарегистрировано в perxis")

        with grpc.insecure_channel("content:9020") as content_channel:
            intercepted_content_channel = grpc.intercept_channel(content_channel, interceptor)

            collections_stub = collections_pb2_grpc.CollectionsStub(intercepted_content_channel)
            roles_stub = roles_pb2_grpc.RolesStub(intercepted_content_channel)
            clients_stub = clients_pb2_grpc.ClientsStub(intercepted_content_channel)
            environments_stub = environments_pb2_grpc.EnvironmentsStub(intercepted_content_channel)

            server = grpc.server(futures.ThreadPoolExecutor(max_workers=10))
            extension_pb2_grpc.add_ExtensionServicer_to_server(
                Servicer(
                    collections_stub, environments_stub, roles_stub, clients_stub
                ), server
            )
            server.add_insecure_port("[::]:50051")
            server.start()
            server.wait_for_termination()


if __name__ == "__main__":
    main()
