package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/users"
	"git.perx.ru/perxis/perxis-go/pkg/users/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestUsersCache(t *testing.T) {

	const (
		userID   = "user_id"
		identity = "user identity"
		size     = 5
		ttl      = 20 * time.Millisecond
	)

	errNotFound := errors.NotFound(errors.New("not found"))

	t.Run("Get from cache", func(t *testing.T) {
		usrs := &mocks.Users{}
		ctx := context.Background()

		svc := CachingMiddleware(cache.NewCache(size, ttl))(usrs)

		usrs.On("Get", mock.Anything, userID).Return(&users.User{ID: userID, Name: "User", Identities: []string{identity}}, nil).Once()

		v1, err := svc.Get(ctx, userID)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, userID)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

		v3, err := svc.GetByIdentity(ctx, identity)
		require.NoError(t, err)
		assert.Same(t, v2, v3, "Ожидается получение объекта из кэша при запросе по Identity.")

		usrs.AssertExpectations(t)
	})

	t.Run("GetByIdentity from cache", func(t *testing.T) {
		usrs := &mocks.Users{}
		ctx := context.Background()

		svc := CachingMiddleware(cache.NewCache(size, ttl))(usrs)

		usrs.On("GetByIdentity", mock.Anything, identity).Return(&users.User{ID: userID, Name: "User", Identities: []string{identity}}, nil).Once()

		v1, err := svc.GetByIdentity(ctx, identity)
		require.NoError(t, err)

		v2, err := svc.GetByIdentity(ctx, identity)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

		v3, err := svc.Get(ctx, userID)
		require.NoError(t, err)
		assert.Same(t, v2, v3, "Ожидается получение объекта из кэша при запросе по userID.")

		usrs.AssertExpectations(t)
	})

	t.Run("Invalidate Cache", func(t *testing.T) {
		t.Run("After Update", func(t *testing.T) {
			usrs := &mocks.Users{}
			ctx := context.Background()

			svc := CachingMiddleware(cache.NewCache(size, ttl))(usrs)

			usrs.On("Get", mock.Anything, userID).Return(&users.User{ID: userID, Name: "User", Identities: []string{identity}}, nil).Once()
			usrs.On("Update", mock.Anything, mock.Anything).Return(nil).Once()

			v1, err := svc.Get(ctx, userID)
			require.NoError(t, err)

			v2, err := svc.GetByIdentity(ctx, identity)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			err = svc.Update(ctx, &users.User{ID: userID, Name: "New User", Identities: []string{identity}})
			require.NoError(t, err)

			usrs.On("GetByIdentity", mock.Anything, identity).Return(&users.User{ID: userID, Name: "New User", Identities: []string{identity}}, nil).Once()

			v3, err := svc.GetByIdentity(ctx, identity)
			require.NoError(t, err)
			assert.NotSame(t, v3, v2, "Ожидается удаление объекта из кеша после обновления и получение его заново из сервиса.")

			v4, err := svc.Get(ctx, userID)
			require.NoError(t, err)
			assert.NotSame(t, v4, v2)
			assert.Same(t, v4, v3, "Ожидается получение нового обьекта из кеша.")

			usrs.AssertExpectations(t)
		})

		t.Run("After Delete", func(t *testing.T) {
			usrs := &mocks.Users{}
			ctx := context.Background()

			svc := CachingMiddleware(cache.NewCache(size, ttl))(usrs)

			usrs.On("Get", mock.Anything, userID).Return(&users.User{ID: userID, Name: "User", Identities: []string{identity}}, nil).Once()
			usrs.On("Delete", mock.Anything, mock.Anything).Return(nil).Once()

			v1, err := svc.Get(ctx, userID)
			require.NoError(t, err)

			v2, err := svc.GetByIdentity(ctx, identity)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			err = svc.Delete(ctx, userID)
			require.NoError(t, err)

			usrs.On("GetByIdentity", mock.Anything, identity).Return(nil, errNotFound).Once()
			usrs.On("Get", mock.Anything, userID).Return(nil, errNotFound).Once()

			_, err = svc.GetByIdentity(ctx, identity)
			require.Error(t, err)
			assert.EqualErrorf(t, err, "not found", "Ожидается удаление объекта из кеша после удаления из хранилища и получение ошибки от сервиса.")

			_, err = svc.Get(ctx, userID)
			require.Error(t, err)
			assert.EqualErrorf(t, err, "not found", "Ожидается удаление объекта из кеша после удаления из хранилища и получение ошибки от сервиса.")

			usrs.AssertExpectations(t)
		})

		t.Run("After TTL expired", func(t *testing.T) {
			usrs := &mocks.Users{}
			ctx := context.Background()

			svc := CachingMiddleware(cache.NewCache(size, ttl))(usrs)

			usrs.On("Get", mock.Anything, userID).Return(&users.User{ID: userID, Name: "User", Identities: []string{identity}}, nil).Once()

			v1, err := svc.Get(ctx, userID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, userID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			time.Sleep(2 * ttl)

			usrs.On("Get", mock.Anything, userID).Return(&users.User{ID: userID, Name: "User", Identities: []string{identity}}, nil).Once()

			v3, err := svc.Get(ctx, userID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v3, "Ожидается получение объекта из кэша при запросе по Identity.")

			usrs.AssertExpectations(t)
		})
	})
}
