package urlsigner

import (
	"net/url"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
)

const secret = "secret_key"

func TestSigner(t *testing.T) {

	t.Run("Not Signed", func(t *testing.T) {
		u, _ := url.Parse("http://example.com/path")

		signer := NewURLSigner(secret, time.Minute, "")
		require.False(t, signer.Check(u))
	})
	t.Run("Simple", func(t *testing.T) {
		u, _ := url.Parse("http://example.com/path")

		signer := NewURLSigner(secret, time.Minute, "")
		u = signer.Sign(u)
		require.True(t, signer.Check(u))
	})
	t.Run("Custom Query Key", func(t *testing.T) {
		u, _ := url.Parse("http://example.com/path")

		signer := NewURLSigner(secret, time.Minute, "custom")
		u = signer.Sign(u)
		require.True(t, signer.Check(u))
	})
	t.Run("URL Expired", func(t *testing.T) {
		u, _ := url.Parse("http://example.com/path")

		signer := NewURLSigner(secret, time.Millisecond, "")
		u = signer.Sign(u)
		time.Sleep(time.Second)
		require.False(t, signer.Check(u))
	})

	t.Run("Required Params", func(t *testing.T) {
		t.Run("Correct", func(t *testing.T) {
			u, _ := url.Parse("http://example.com/path?param1=1&param2=2")
			signer := NewURLSigner(secret, time.Minute, "", "param1", "param2")
			u = signer.Sign(u)
			require.True(t, signer.Check(u))
		})
		t.Run("One Param Empty", func(t *testing.T) {
			u, _ := url.Parse("http://example.com/path?param1=1")
			signer := NewURLSigner(secret, time.Minute, "", "param1", "param2")
			u = signer.Sign(u)
			require.True(t, signer.Check(u))
		})
		t.Run("Exchanged Values", func(t *testing.T) {
			u, _ := url.Parse("http://example.com/path?param1=1&param2=2")
			signer := NewURLSigner(secret, time.Minute, "", "param1", "param2")
			u = signer.Sign(u)

			q := u.Query()
			q.Set("param1", "2")
			q.Set("param2", "1")
			u.RawQuery = q.Encode()

			require.False(t, signer.Check(u))
		})
	})
	t.Run("Extra params", func(t *testing.T) {
		u, _ := url.Parse("http://example.com/path?param1=1")
		signer := NewURLSigner(secret, time.Minute, "", "param1")
		u = signer.Sign(u)

		q := u.Query()
		q.Set("extra", "100")
		u.RawQuery = q.Encode()

		require.True(t, signer.Check(u))
	})
	t.Run("Array Param Value", func(t *testing.T) {
		u, _ := url.Parse("http://example.com/path?param1=1&param1=2&param1=3")
		signer := NewURLSigner(secret, time.Minute, "", "param1")
		u = signer.Sign(u)

		q := u.Query()
		q.Set("param1", "1")
		u.RawQuery = q.Encode()

		require.False(t, signer.Check(u))
	})
}
