package spaces

import (
	"context"

	"github.com/pkg/errors"
)

// @microgen grpc
// @protobuf git.perx.ru/perxis/perxis-go/proto/spaces
// @grpc-addr content.spaces.Spaces
type Spaces interface {
	// Create - создает пространство для размещения контента
	// В процессе создание пространства выполняются следующие шаги:
	// 1. Создается запись о пространстве
	// 2. Создается БД пространства
	// 3. Создается master окружение
	// 4. Создаются необходимые схема согласно указанной конфигурации
	// 5. Выполняется копирование данных согласно указанной конфигурации
	// Шаги могут выполняться в отложенном режиме, при этом пространство может быть недоступно для
	// выполнения любых операций. Попытка обращения к пространству будет сопровождаться ошибкой
	// ErrUnavailable

	Create(ctx context.Context, space *Space) (created *Space, err error)
	Get(ctx context.Context, spaceId string) (space *Space, err error)
	List(ctx context.Context, orgId string) (spaces []*Space, err error)
	Update(ctx context.Context, space *Space) (err error)
	UpdateConfig(ctx context.Context, spaceId string, config *Config) (err error)
	Delete(ctx context.Context, spaceId string) (err error)

	// Transfer устанавливает для пространства значение поля RequestedMoveTo. После этого пространство
	// будет отображаться в списке входящих запросов на перемещение в организации `orgID` (запрос ListIncoming)
	// С пространством можно продолжать работу в текущей организации, пока запрос на перемещение не будет
	// принят в целевой организации
	// Перенос может быть инициирован только владельцем организации, которой принадлежит пространство
	Transfer(ctx context.Context, spaceID, transferToOrg string) (err error)

	// AbortTransfer - отменить перемещение пространства в другую организацию. Может быть вызван как
	// владельцем пространства-инициатора, там и владельцем принимающего пространства
	AbortTransfer(ctx context.Context, spaceID string) (err error)

	// ListTransfers возвращает список пространств, перемещение которых было запрошено в текущую организацию
	ListTransfers(ctx context.Context, orgID string) (spaces []*Space, err error)

	// Move - перенести пространство в организацию, установленную в Space.TransferToOrg. Пространство переносится со
	// всеми входящими в него данными: ролями, участниками, контентом, пр. и более не будет доступно в
	// исходной организации.
	// В случае, если запрос осуществляется с системным уровнем доступа и передан параметр `orgID`, то перенос будет
	// осуществлен вне зависимости от того, был он инициирован через метод `Transfer` или нет
	Move(ctx context.Context, spaceID, orgID string) (err error)
}

func IsSpaceAvailable(ctx context.Context, spcs Spaces, spaceId string) error {
	spc, err := spcs.Get(ctx, spaceId)

	if err != nil {
		return errors.Wrap(err, "space not available")
	}

	if spc.State != StateReady {
		return errors.New("space not available")
	}

	return nil
}
