package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/errors"

	"git.perx.ru/perxis/perxis-go/pkg/spaces"
	spmocks "git.perx.ru/perxis/perxis-go/pkg/spaces/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestRolesCache(t *testing.T) {

	const (
		spaceID = "spaceID"
		orgID   = "orgID"
		size    = 5
		ttl     = 20 * time.Millisecond
	)

	errNotFound := errors.NotFound(errors.New("not found"))

	ctx := context.Background()

	t.Run("Get from cache", func(t *testing.T) {
		sp := &spmocks.Spaces{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(sp)

		sp.On("Get", mock.Anything, spaceID).Return(&spaces.Space{ID: spaceID, OrgID: orgID, Name: "Space"}, nil).Once()

		v1, err := svc.Get(ctx, spaceID)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, spaceID)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается при повторном запросе получение объекта из кэша.")

		sp.AssertExpectations(t)
	})

	t.Run("List from cache", func(t *testing.T) {
		sp := &spmocks.Spaces{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(sp)

		sp.On("List", mock.Anything, orgID).Return([]*spaces.Space{{ID: spaceID, OrgID: orgID, Name: "Space"}}, nil).Once()

		vl1, err := svc.List(ctx, orgID)
		require.NoError(t, err)

		vl2, err := svc.List(ctx, orgID)
		require.NoError(t, err)
		assert.Same(t, vl1[0], vl2[0], "Ожидается при повторном запросе получение объектов из кэша.")

		sp.AssertExpectations(t)
	})

	t.Run("Invalidate cache", func(t *testing.T) {
		t.Run("After Update", func(t *testing.T) {
			sp := &spmocks.Spaces{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(sp)

			sp.On("Get", mock.Anything, spaceID).Return(&spaces.Space{ID: spaceID, OrgID: orgID, Name: "Space"}, nil).Once()
			sp.On("List", mock.Anything, orgID).Return([]*spaces.Space{{ID: spaceID, OrgID: orgID, Name: "Space"}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается при повторном запросе получение объекта из кэша.")

			vl1, err := svc.List(ctx, orgID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, orgID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается при повторном запросе получение объектов из кэша.")

			sp.On("Update", mock.Anything, mock.Anything).Return(nil).Once()

			err = svc.Update(ctx, &spaces.Space{ID: spaceID, OrgID: orgID, Name: "SpaceUPD"})
			require.NoError(t, err)

			sp.On("Get", mock.Anything, spaceID).Return(&spaces.Space{ID: spaceID, OrgID: orgID, Name: "SpaceUPD"}, nil).Once()
			sp.On("List", mock.Anything, orgID).Return([]*spaces.Space{{ID: spaceID, OrgID: orgID, Name: "SpaceUPD"}}, nil).Once()

			v3, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v3, "Ожидается что кеш объекта был удален после обновления объекта.")

			vl3, err := svc.List(ctx, orgID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидается что кеш объектов был удален после обновления объекта.")

			sp.AssertExpectations(t)
		})

		t.Run("After UpdateConfig", func(t *testing.T) {
			sp := &spmocks.Spaces{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(sp)

			sp.On("Get", mock.Anything, spaceID).Return(&spaces.Space{ID: spaceID, OrgID: orgID, Name: "Space"}, nil).Once()
			sp.On("List", mock.Anything, orgID).Return([]*spaces.Space{{ID: spaceID, OrgID: orgID, Name: "Space"}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается при повторном запросе получение объекта из кэша.")

			vl1, err := svc.List(ctx, orgID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, orgID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается при повторном запросе получение объектов из кэша.")

			sp.On("UpdateConfig", mock.Anything, spaceID, mock.Anything).Return(nil).Once()

			err = svc.UpdateConfig(ctx, spaceID, &spaces.Config{Features: []string{"feature"}})
			require.NoError(t, err)

			sp.On("Get", mock.Anything, spaceID).Return(&spaces.Space{ID: spaceID, OrgID: orgID, Name: "SpaceUPD", Config: &spaces.Config{Features: []string{"feature"}}}, nil).Once()
			sp.On("List", mock.Anything, orgID).Return([]*spaces.Space{{ID: spaceID, OrgID: orgID, Name: "SpaceUPD", Config: &spaces.Config{Features: []string{"feature"}}}}, nil).Once()

			v3, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v3, "Ожидается что кеш объекта был удален после обновления объекта.")

			vl3, err := svc.List(ctx, orgID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидается что кеш объектов был удален после обновления объекта.")

			sp.AssertExpectations(t)
		})

		t.Run("After Delete", func(t *testing.T) {
			sp := &spmocks.Spaces{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(sp)

			sp.On("Get", mock.Anything, spaceID).Return(&spaces.Space{ID: spaceID, OrgID: orgID, Name: "Space"}, nil).Once()
			sp.On("List", mock.Anything, orgID).Return([]*spaces.Space{{ID: spaceID, OrgID: orgID, Name: "Space"}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается при повторном запросе получение объекта из кэша.")

			vl1, err := svc.List(ctx, orgID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, orgID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается при повторном запросе получение объектов из кэша.")

			sp.On("Delete", mock.Anything, spaceID).Return(nil).Once()

			err = svc.Delete(ctx, spaceID)
			require.NoError(t, err)

			sp.On("Get", mock.Anything, spaceID).Return(nil, errNotFound).Once()
			sp.On("List", mock.Anything, orgID).Return([]*spaces.Space{}, nil).Once()

			_, err = svc.Get(ctx, spaceID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается что после удаления объекта кеш  был удален и получена ошибка от сервиса.")

			vl3, err := svc.List(ctx, orgID)
			require.NoError(t, err)
			assert.Len(t, vl3, 0, "Ожидается что после удаления кеш объектов был удален.")

			sp.AssertExpectations(t)
		})

		t.Run("After Create", func(t *testing.T) {
			sp := &spmocks.Spaces{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(sp)

			sp.On("List", mock.Anything, orgID).Return([]*spaces.Space{{ID: spaceID, OrgID: orgID, Name: "Space"}}, nil).Once()

			vl1, err := svc.List(ctx, orgID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, orgID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается при повторном запросе получение объектов из кэша.")

			sp.On("Create", mock.Anything, mock.Anything).Return(&spaces.Space{ID: "spaceID2", OrgID: orgID, Name: "Space2"}, nil).Once()

			_, err = svc.Create(ctx, &spaces.Space{ID: "spaceID2", OrgID: orgID, Name: "Space2"})
			require.NoError(t, err)

			sp.On("List", mock.Anything, orgID).Return([]*spaces.Space{{ID: spaceID, OrgID: orgID, Name: "Space"}, {ID: "spaceID2", OrgID: orgID, Name: "Space2"}}, nil).Once()

			vl3, err := svc.List(ctx, orgID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидается что кеш объектов был удален после создания нового объекта.")

			sp.AssertExpectations(t)
		})

		t.Run("After TTL expired", func(t *testing.T) {
			sp := &spmocks.Spaces{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(sp)

			sp.On("Get", mock.Anything, spaceID).Return(&spaces.Space{ID: spaceID, OrgID: orgID, Name: "Space"}, nil).Once()

			v1, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается при повторном запросе получение объекта из кэша.")

			time.Sleep(2 * ttl)
			sp.On("Get", mock.Anything, spaceID).Return(&spaces.Space{ID: spaceID, OrgID: orgID, Name: "Space"}, nil).Once()

			v3, err := svc.Get(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v3, "Ожидается удаление объекта из кэша по истечению ttl.")

			sp.AssertExpectations(t)
		})
	})
}
