package setup

import (
	"context"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/data"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/permission"
	"git.perx.ru/perxis/perxis-go/pkg/roles"
	"go.uber.org/zap"
)

var (
	ErrCheckRoles     = errors.New("role check error")
	ErrInstallRoles   = errors.New("failed to install role")
	ErrUninstallRoles = errors.New("failed to uninstall role")
)

type RolesOption func(c *RoleConfig)
type UpdateRoleFn func(s *Setup, exist, new *roles.Role) (*roles.Role, bool)
type DeleteRoleFn func(s *Setup, role *roles.Role) bool

type RoleConfig struct {
	role     *roles.Role
	UpdateFn UpdateRoleFn
	DeleteFn DeleteRoleFn
}

func NewRoleConfig(role *roles.Role, opt ...RolesOption) RoleConfig {
	c := RoleConfig{role: role}

	UpdateExistingRole()(&c)
	DeleteRoleIfRemove()(&c)

	for _, o := range opt {
		o(&c)
	}
	return c
}

func OverwriteRole() RolesOption {
	return func(c *RoleConfig) {
		c.UpdateFn = func(s *Setup, old, new *roles.Role) (*roles.Role, bool) { return new, true }
	}
}

//func OverwriteRoleIfChanged() RolesOption {
//	return func(c *RoleConfig) {
//		c.UpdateFn = func(s *Setup, old, new *roles.Role) (*roles.Role, bool) {
//			changed := old.Description != new.Description || old.AllowManagement != new.AllowManagement ||
//				!util.ElementsMatch(old.Environments, new.Environments)
//			return new, changed
//		}
//	}
//}

func KeepExistingRole() RolesOption {
	return func(c *RoleConfig) {
		c.UpdateFn = func(s *Setup, old, new *roles.Role) (*roles.Role, bool) { return old, false }
	}
}

func DeleteRole() RolesOption {
	return func(c *RoleConfig) {
		c.DeleteFn = func(s *Setup, role *roles.Role) bool { return true }
	}
}

func DeleteRoleIfRemove() RolesOption {
	return func(c *RoleConfig) {
		c.DeleteFn = func(s *Setup, role *roles.Role) bool { return s.IsRemove() }
	}
}

func UpdateExistingRole() RolesOption {
	return func(c *RoleConfig) {
		c.UpdateFn = func(s *Setup, exist, new *roles.Role) (*roles.Role, bool) {

			// если передан флаг force, то обновляем все поля роли на переданные
			if s.IsForce() {
				return new, true
			}

			if exist.Description == "" {
				exist.Description = new.Description
			}

			if len(exist.Environments) == 0 {
				exist.Environments = new.Environments
			}

			if !data.Contains(s.EnvironmentID, exist.Environments) {
				exist.Environments = append(exist.Environments, s.EnvironmentID)
			}

			exist.Rules = permission.MergeRules(exist.Rules, new.Rules)

			if !exist.AllowManagement {
				exist.AllowManagement = new.AllowManagement
			}

			return exist, true
		}
	}
}

func (s *Setup) InstallRoles(ctx context.Context) error {
	if len(s.Roles) == 0 {
		return nil
	}

	s.logger.Debug("Install role", zap.String("Space ID", s.SpaceID), zap.Int("Roles", len(s.Roles)))

	for _, c := range s.Roles {
		if err := s.InstallRole(ctx, c); err != nil {
			s.logger.Error("Failed to install role", zap.String("Role ID", c.role.ID), zap.Error(err))
			return errors.Wrap(errors.WithDetailf(err, "Возникла ошибка при настройке роли %s(%s)", c.role.ID, c.role.Description), "failed to install role")
		}
	}

	return nil
}

func (s *Setup) InstallRole(ctx context.Context, c RoleConfig) error {
	role := c.role
	role.SpaceID = s.SpaceID

	if !data.Contains(s.EnvironmentID, c.role.Environments) {
		role.Environments = append(role.Environments, s.EnvironmentID)
	}

	exist, err := s.content.Roles.Get(ctx, s.SpaceID, role.ID)
	if err != nil {
		if !strings.Contains(err.Error(), roles.ErrNotFound.Error()) {
			return err
		}

		_, err = s.content.Roles.Create(ctx, role)
		return err
	}

	if r, upd := c.UpdateFn(s, exist, role); upd {
		return s.content.Roles.Update(ctx, r)
	}

	return nil
}

func (s *Setup) UninstallRoles(ctx context.Context) error {
	if len(s.Roles) == 0 {
		return nil
	}

	s.logger.Debug("Uninstall role", zap.String("Space ID", s.SpaceID), zap.Int("Roles", len(s.Roles)))

	for _, c := range s.Roles {
		if err := s.UninstallRole(ctx, c); err != nil {
			s.logger.Error("Failed to uninstall role", zap.String("Role ID", c.role.ID), zap.Error(err))
			return errors.WithDetailf(errors.Wrap(err, "failed to uninstall role"), "Возникла ошибка при удалении роли %s(%s)", c.role.ID, c.role.Description)
		}
	}

	return nil
}

func (s *Setup) UninstallRole(ctx context.Context, c RoleConfig) error {
	if c.DeleteFn(s, c.role) {
		err := s.content.Roles.Delete(ctx, s.SpaceID, c.role.ID)
		if err != nil && !strings.Contains(err.Error(), roles.ErrNotFound.Error()) {
			return err
		}
	}
	return nil
}

func (s *Setup) CheckRoles(ctx context.Context) error {
	if len(s.Roles) == 0 {
		return nil
	}

	s.logger.Debug("Check role", zap.String("Space ID", s.SpaceID))

	var errs []error
	for _, c := range s.Roles {
		if err := s.CheckRole(ctx, c.role); err != nil {
			errs = append(errs, errors.WithDetailf(err, "Не найдена роль %s(%s)", c.role.ID, c.role.Description))
		}
	}

	if len(errs) > 0 {
		return errors.WithErrors(ErrCheckRoles, errs...)
	}

	return nil
}

func (s *Setup) CheckRole(ctx context.Context, role *roles.Role) error {
	_, err := s.content.Roles.Get(ctx, s.SpaceID, role.ID)
	return err
}
