package setup

import (
	"context"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/content"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/items"
	itemsMock "git.perx.ru/perxis/perxis-go/pkg/items/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestItem_OverwriteFields(t *testing.T) {
	tests := []struct {
		name    string
		fields  []string
		new     *items.Item
		old     *items.Item
		want    *items.Item
		changed bool
	}{
		{
			name:    "Empty",
			fields:  []string{},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			changed: false,
		},
		{
			name:    "Not found",
			fields:  []string{"notfound"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			changed: false,
		},
		{
			name:    "Equal value",
			fields:  []string{"key"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			changed: false,
		},
		{
			name:    "Not Equal value #1",
			fields:  []string{"key"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value1"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value", "key2": "value2"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"key": "value1", "key2": "value2"}},
			changed: true,
		},
		{
			name:    "Not Equal value #2",
			fields:  []string{"key", "key2"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value1", "key2": "value2"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value", "key3": "value3"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"key": "value1", "key2": "value2", "key3": "value3"}},
			changed: true,
		},
		{
			name:    "Equal nested",
			fields:  []string{"hoop.exclude"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": true}}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": true}, "key": "value"}},
			changed: false,
		},
		{
			name:    "Not Equal nested",
			fields:  []string{"hoop.exclude"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": true}}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": false}, "key": "value"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": true}, "key": "value"}},
			changed: true,
		},
		{
			name:    "System",
			fields:  []string{"id"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"id": "value"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"id": "value"}},
			changed: false,
		},
	}
	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			c := ItemConfig{item: test.new}
			OverwriteFields(test.fields...)(&c)

			got, changed := c.UpdateFn(nil, test.old, test.new)
			require.Equal(t, test.changed, changed)
			if !test.changed {
				return
			}
			assert.Equal(t, test.want, got)

		})
	}
}

func TestItem_KeepFields(t *testing.T) {
	tests := []struct {
		name    string
		fields  []string
		new     *items.Item
		old     *items.Item
		want    *items.Item
		changed bool
	}{
		{
			name:    "Empty",
			fields:  []string{},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "0"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			changed: true,
		},
		{
			name:    "Not found",
			fields:  []string{"notfound"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "0"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			changed: true,
		},
		{
			name:    "Equal value",
			fields:  []string{"key"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value", "key2": "value2"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"key": "value", "key2": "value2"}},
			changed: true,
		},
		{
			name:    "Not Equal value #1",
			fields:  []string{"key"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value1", "key3": "value3"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value", "key2": "value2"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"key": "value", "key3": "value3"}},
			changed: true,
		},
		{
			name:    "Not Equal value #2",
			fields:  []string{"key", "key2"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value1", "key3": "value3"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"key": "value", "key2": "value2"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"key": "value", "key2": "value2", "key3": "value3"}},
			changed: true,
		},
		{
			name:    "Equal nested",
			fields:  []string{"hoop.exclude"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": true}}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": true}, "key": "value"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": true}}},
			changed: true,
		},
		{
			name:    "Not Equal nested",
			fields:  []string{"hoop.exclude"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": true}}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": false}, "key": "value"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"hoop": map[string]interface{}{"exclude": false}}},
			changed: true,
		},
		{
			name:    "System",
			fields:  []string{"id"},
			new:     &items.Item{ID: "id", Data: map[string]interface{}{"id": "value1"}},
			old:     &items.Item{ID: "id", Data: map[string]interface{}{"id": "value"}},
			want:    &items.Item{ID: "id", Data: map[string]interface{}{"id": "value1"}},
			changed: true,
		},
	}
	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			c := ItemConfig{item: test.new}
			KeepFields(test.fields...)(&c)

			got, changed := c.UpdateFn(nil, test.old, test.new)
			require.Equal(t, test.changed, changed)
			if !test.changed {
				return
			}
			assert.Equal(t, test.want, got)

		})
	}
}

func TestSetup_InstallItems(t *testing.T) {

	tests := []struct {
		name      string
		items     []*items.Item
		itemsCall func(svc *itemsMock.Items)
		wantErr   func(t *testing.T, err error)
	}{
		{
			name:      "Nil clients",
			items:     nil,
			itemsCall: func(svc *itemsMock.Items) {},
			wantErr: func(t *testing.T, err error) {
				assert.NoError(t, err)
			},
		},
		{
			name:  "Install one item success",
			items: []*items.Item{{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}},
			itemsCall: func(svc *itemsMock.Items) {
				svc.On("Find", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil, 0, nil).Once()
				svc.On("Create", mock.Anything, &items.Item{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}).Return(&items.Item{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}, nil).Once()
				svc.On("Publish", mock.Anything, &items.Item{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}).Return(nil).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.NoError(t, err)
			},
		},
		{
			name:  "Install one item fails",
			items: []*items.Item{{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}},
			itemsCall: func(svc *itemsMock.Items) {
				svc.On("Find", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil, 0, nil).Once()
				svc.On("Create", mock.Anything, &items.Item{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}).Return(nil, errors.New("some error")).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.Error(t, err)
				assert.EqualError(t, err, "failed to install item: create item: some error")
				assert.Contains(t, errors.GetDetail(err), "Возникла ошибка при добавлении элемента 1(coll)")
			},
		},
		{
			name: "Install multiple items success",
			items: []*items.Item{
				{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}},
				{ID: "2", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}},
			},
			itemsCall: func(svc *itemsMock.Items) {
				svc.On("Find", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil, 0, nil).Once()
				svc.On("Create", mock.Anything, &items.Item{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}).Return(&items.Item{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}, nil).Once()
				svc.On("Create", mock.Anything, &items.Item{ID: "2", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}).Return(&items.Item{ID: "2", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}, nil).Once()
				svc.On("Publish", mock.Anything, &items.Item{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}).Return(nil).Once()
				svc.On("Publish", mock.Anything, &items.Item{ID: "2", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}).Return(nil).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.NoError(t, err)
			},
		},
		{
			name: "Install multiple items fails",
			items: []*items.Item{
				{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}},
				{ID: "2", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}},
			},
			itemsCall: func(svc *itemsMock.Items) {
				svc.On("Find", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil, 0, nil).Once()
				svc.On("Create", mock.Anything, &items.Item{ID: "1", SpaceID: "sp", EnvID: "env", CollectionID: "coll", Data: map[string]interface{}{"text": "test"}}).Return(nil, errors.New("some error")).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.Error(t, err)
				assert.EqualError(t, err, "failed to install item: create item: some error")
				assert.Contains(t, errors.GetDetail(err), "Возникла ошибка при добавлении элемента 1(coll)")
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			i := &itemsMock.Items{}
			if tt.itemsCall != nil {
				tt.itemsCall(i)
			}

			s := NewSetup(&content.Content{Items: i}, "sp", "env", nil)
			s.AddItems(tt.items)
			tt.wantErr(t, s.InstallItems(context.Background()))
		})
	}
}
