package setup

import (
	"context"
	"reflect"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/items"
	"go.uber.org/zap"
)

var (
	ErrCheckItems     = errors.New("items check error")
	ErrInstallItems   = errors.New("failed to install items")
	ErrUninstallItems = errors.New("failed to uninstall items")
	ErrItemsNotFound  = errors.New("item not found")
)

type ItemsOption func(c *ItemConfig)
type UpdateItemFn func(s *Setup, exist, new *items.Item) (*items.Item, bool)
type DeleteItemFn func(s *Setup, col *items.Item) bool

type ItemConfig struct {
	item     *items.Item
	UpdateFn UpdateItemFn
	DeleteFn DeleteItemFn
}

func NewItemConfig(item *items.Item, opt ...ItemsOption) ItemConfig {
	c := ItemConfig{item: item}

	KeepExistingItem()(&c)
	DeleteItemIfRemove()(&c)

	for _, o := range opt {
		o(&c)
	}

	return c
}

func OverwriteItem() ItemsOption {
	return func(c *ItemConfig) {
		c.UpdateFn = func(s *Setup, old, new *items.Item) (*items.Item, bool) { return new, true }
	}
}

func OverwriteFields(fields ...string) ItemsOption {
	return func(c *ItemConfig) {
		c.UpdateFn = func(s *Setup, old, new *items.Item) (*items.Item, bool) {

			var changed bool
			for _, field := range fields {
				if items.IsSystemField(field) {
					continue
				}

				newValue, err := new.Get(field)
				if err != nil {
					continue
				}

				oldValue, err := old.Get(field)
				if err != nil || newValue != oldValue {
					changed = true
					if err = old.Set(field, newValue); err != nil {
						return nil, false // не обновляем данные если не удалось установить значение
					}
				}
			}

			return old, changed
		}
	}
}

func KeepFields(fields ...string) ItemsOption {
	return func(c *ItemConfig) {
		c.UpdateFn = func(s *Setup, old, new *items.Item) (*items.Item, bool) {

			for _, field := range fields {
				if items.IsSystemField(field) {
					continue
				}

				oldValue, err := old.Get(field)
				if err != nil {
					continue
				}

				newValue, err := new.Get(field)
				if err != nil || newValue != oldValue {
					if err = new.Set(field, oldValue); err != nil {
						return nil, false // не обновляем данные если не удалось установить значение
					}
				}
			}

			return new, !reflect.DeepEqual(old, new)
		}
	}
}

func KeepExistingItem() ItemsOption {
	return func(c *ItemConfig) {
		c.UpdateFn = func(s *Setup, old, new *items.Item) (*items.Item, bool) { return old, false }
	}
}

func DeleteItem() ItemsOption {
	return func(c *ItemConfig) {
		c.DeleteFn = func(s *Setup, item *items.Item) bool { return true }
	}
}

func DeleteItemIfRemove() ItemsOption {
	return func(c *ItemConfig) {
		c.DeleteFn = func(s *Setup, item *items.Item) bool { return s.IsRemove() }
	}
}

func (s *Setup) InstallItems(ctx context.Context) error {
	if len(s.Items) == 0 {
		return nil
	}

	s.logger.Debug("Install items", zap.Int("Items", len(s.Items)))

	for col, itms := range s.groupByCollection() {
		exists, err := s.getExisting(ctx, col, itms)
		if err != nil {
			return err
		}
		for _, c := range itms {
			if err := s.InstallItem(ctx, exists, c); err != nil {
				s.logger.Error("Failed to install item",
					zap.String("ID", c.item.ID),
					zap.String("Collection", c.item.CollectionID),
					zap.Error(err),
				)
				return errors.WithDetailf(errors.Wrap(err, "failed to install item"), "Возникла ошибка при добавлении элемента %s(%s)", c.item.ID, c.item.CollectionID)
			}
		}
	}

	return nil
}

func (s *Setup) InstallItem(ctx context.Context, exists map[string]*items.Item, c ItemConfig) error {
	item := c.item
	item.SpaceID, item.EnvID = s.SpaceID, s.EnvironmentID

	exist, ok := exists[item.ID]
	if !ok {
		return items.CreateAndPublishItem(ctx, s.content.Items, item)
	}

	if item, changed := c.UpdateFn(s, exist, c.item); changed {
		return items.UpdateAndPublishItem(ctx, s.content.Items, item)
	}

	return nil
}

func (s *Setup) UninstallItems(ctx context.Context) error {
	if len(s.Items) == 0 {
		return nil
	}

	s.logger.Debug("Uninstall items", zap.Int("Items", len(s.Items)))

	for _, c := range s.Items {
		if err := s.UninstallItem(ctx, c); err != nil {
			s.logger.Error("Failed to uninstall item",
				zap.String("Item", c.item.ID),
				zap.String("Item", c.item.CollectionID),
				zap.Error(err),
			)
			return errors.WithDetailf(errors.Wrap(err, "failed to uninstall item"), "Возникла ошибка при удалении элемента %s(%s)", c.item.ID, c.item.CollectionID)
		}
	}

	return nil
}

func (s *Setup) UninstallItem(ctx context.Context, c ItemConfig) error {
	if c.DeleteFn(s, c.item) {
		err := s.content.Items.Delete(ctx, &items.Item{SpaceID: s.SpaceID, EnvID: s.EnvironmentID, CollectionID: c.item.CollectionID, ID: c.item.ID})
		if err != nil && !strings.Contains(err.Error(), items.ErrNotFound.Error()) {
			return err
		}
	}
	return nil
}

func (s *Setup) CheckItems(ctx context.Context) error {
	if len(s.Items) == 0 {
		return nil
	}

	var errs []error
	s.logger.Debug("Check items", zap.Int("Items", len(s.Items)))

	for col, itms := range s.groupByCollection() {
		exists, err := s.getExisting(ctx, col, itms)
		if err != nil {
			return err
		}

		for _, c := range itms {
			if _, ok := exists[c.item.ID]; !ok {
				errs = append(errs, errors.WithDetailf(errors.New("not found"), "Не найден элемент %s(%s)", c.item.ID, c.item.CollectionID))
			}
		}
	}

	if len(errs) > 0 {
		return errors.WithErrors(ErrCheckItems, errs...)
	}

	return nil
}

func (s *Setup) removeItems(collID string) {
	itms := make([]ItemConfig, 0, len(s.Items))
	for _, i := range s.Items {
		if i.item.CollectionID != collID {
			itms = append(itms, i)
		}
	}
	s.Items = itms
}

func (s *Setup) groupByCollection() map[string][]ItemConfig {
	itemsByColl := map[string][]ItemConfig{}
	for _, i := range s.Items {
		cfg, ok := itemsByColl[i.item.CollectionID]
		if !ok {
			itemsByColl[i.item.CollectionID] = []ItemConfig{i}
			continue
		}
		itemsByColl[i.item.CollectionID] = append(cfg, i)
	}
	return itemsByColl
}

func (s *Setup) getExisting(ctx context.Context, collID string, configs []ItemConfig) (map[string]*items.Item, error) {
	itms, _, err := s.content.Items.Find(
		ctx,
		s.SpaceID,
		s.EnvironmentID,
		collID,
		&items.Filter{ID: getItemIds(configs)},
		&items.FindOptions{Regular: true, Hidden: true, Templates: true},
	)
	if err != nil {
		s.logger.Error("Failed to find existing items",
			zap.String("Collection", collID),
			zap.Error(err),
		)
		return nil, errors.WithDetailf(errors.Wrap(err, "failed to find existing items"), "Возникла ошибка при поиске элементов в коллекции %s", collID)
	}

	exists := make(map[string]*items.Item, len(itms))
	for _, itm := range itms {
		exists[itm.ID] = itm
	}
	return exists, nil
}

func getItemIds(items []ItemConfig) []string {
	var ids []string
	for _, i := range items {
		ids = append(ids, i.item.ID)
	}
	return ids
}
