package schemaloader

import (
	"context"
	"net/url"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/collections"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
)

// NewLoader возвращает новый загрузчик схем из коллекций
// используется только на сервере
// на клиенте нужно использовать методы получения полностью загруженных схем, для которых разрешение происходит на сервере
func NewLoader(svc collections.Collections) field.Loader {
	return &loader{svc: svc}
}

type loader struct {
	svc collections.Collections
}

// Load - возвращает поля по референсу из коллекций (не загруженные)
func (l *loader) Load(ctx context.Context, ref string) ([]*field.Field, error) {
	spaceID, envID, colID, err := parseRef(ctx, ref)
	if err != nil {
		return nil, err
	}

	filter := &collections.Filter{ID: []string{colID}}

	collections, err := l.svc.List(ctx, spaceID, envID, filter)
	if err != nil {
		return nil, errors.Wrapf(err, "schemaloader: failed to get collections for \"%s\"", ref)
	}

	var schemas []*field.Field
	for _, s := range collections {
		if s.Schema != nil {
			schemas = append(schemas, &s.Schema.Field)
		}
	}

	if len(schemas) == 0 {
		return nil, errors.Errorf("schema not found \"%s\"", ref)
	}

	return schemas, nil
}

func parseRef(ctx context.Context, ref string) (spaceID, envID, colID string, err error) {
	var u *url.URL
	if u, err = url.Parse(ref); err != nil {
		return
	}

	parts := strings.SplitN(u.Path, "/", 3)

	switch len(parts) {
	case 1:
		colID = parts[0]
	case 2:
		spaceID = parts[0]
		envID = "master"
		colID = parts[1]
	case 3:
		spaceID = parts[0]
		envID = parts[1]
		colID = parts[2]
	}

	if colID == "" {
		err = errors.Errorf("invalid schema reference \"%s\"", ref)
	}

	if loaderCtx := GetContext(ctx); loaderCtx != nil {
		if spaceID == "" {
			spaceID = loaderCtx.SpaceID
		}

		if envID == "" {
			envID = loaderCtx.EnvID
		}
	}

	if spaceID == "" {
		err = errors.Errorf("can't identify space for reference \"%s\"", ref)
	}

	return
}
