package walk

import (
	"context"
	"fmt"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
)

// DataFunc тип для функции обработки данных
type DataFunc func(c *WalkContext) error

// FieldConfig описывает какие действия с полем нужно предпринять при обходе данных
type FieldConfig struct {
	Fn DataFunc // Пользовательская функция обработки данных поля
}

// WalkConfig настройки обхода данных
type WalkConfig struct {
	// Настройки для полей, в качестве ключа указывается абсолютный путь поля
	// Например: "a.b.c.1" (числа для slice)
	Fields map[string]FieldConfig
}

// Walker позволяет выполнять обход данных для соответствующей схемы
type Walker struct {
	schema    *schema.Schema
	config    *WalkConfig
	DefaultFn DataFunc // Функция обработки данных применяемая по умолчанию
}

// NewWalker создает экземпляр
func NewWalker(schema *schema.Schema, config *WalkConfig) *Walker {
	return &Walker{
		schema:    schema,
		config:    config,
		DefaultFn: GenericMerge,
	}
}

// WalkContext контекст обхода данных
type WalkContext struct {
	Ctx     context.Context
	Path    string       // Путь к родительским данным
	Key     interface{}  // Ключ или индекс текущих данных
	Field   *field.Field // Поля схемы соответсвующее текущим данным
	Dst     interface{}  // Данные приемника
	Src     interface{}  // Данные источника
	Changed bool         // Флаг показывающий, что данные приемника изменились
}

// GetPath возвращает путь соответсвующий текущему контексту
func (w WalkContext) GetPath(keys ...interface{}) string {
	p := make([]string, 0, 10)

	if w.Path != "" {
		p = append(p, w.Path)
	}

	if w.Key != nil {
		p = append(p, fmt.Sprintf("%v", w.Key))
	}

	for _, k := range keys {
		p = append(p, fmt.Sprintf("%v", k))
	}

	return strings.Join(p, ".")
}

// Clone создает копию контекста
func (w WalkContext) Clone() *WalkContext {
	return &w
}

// DataWalk выполняет обход данных и возвращает измененные данные
func (m *Walker) DataWalk(ctx context.Context, dst, src interface{}) (res interface{}, changed bool, err error) {
	wc := &WalkContext{
		Ctx:   ctx,
		Field: &m.schema.Field,
		Dst:   dst,
		Src:   src,
	}

	err = m.datawalk(wc)

	return wc.Dst, wc.Changed, err
}

func (m *Walker) datawalk(w *WalkContext) (err error) {
	path := w.GetPath()
	fn := m.DefaultFn

	fieldCfg, _ := m.config.Fields[path]

	if fieldCfg.Fn != nil {
		fn = fieldCfg.Fn
	}

	if err = fn(w); err != nil {
		return
	}

	switch p := w.Field.Params.(type) {

	case *field.ObjectParameters:
		d, _ := w.Dst.(map[string]interface{})
		s, _ := w.Src.(map[string]interface{})
		res := make(map[string]interface{})

		keys := make(map[string]struct{})
		for k := range d {
			keys[k] = struct{}{}
		}
		for k := range s {
			keys[k] = struct{}{}
		}

		fields := p.GetFields(true)

		for k := range keys {
			f, ok := fields[k]
			if !ok {
				continue
			}

			wc := WalkContext{
				Ctx:   w.Ctx,
				Path:  w.GetPath(),
				Key:   k,
				Field: f,
				Dst:   d[k],
				Src:   s[k],
			}

			if err = m.datawalk(&wc); err != nil {
				return
			}

			if wc.Dst != nil {
				res[k] = wc.Dst
			}

			if wc.Changed {
				w.Changed = true
			}
		}
		if len(res) > 0 {
			w.Dst = res
		}

	case *field.ArrayParameters:
		d, _ := w.Dst.([]interface{})
		s, _ := w.Src.([]interface{})
		for i, v := range d {
			var src_v interface{}
			if i < len(s) {
				src_v = s[i]
			}
			wc := WalkContext{
				Ctx:   w.Ctx,
				Path:  w.GetPath(),
				Key:   i,
				Field: p.Item,
				Dst:   v,
				Src:   src_v,
			}
			if err = m.datawalk(&wc); err != nil {
				return
			}
			if wc.Changed {
				d[i] = wc.Dst
				w.Changed = true
			}
		}

	}

	return
}
