package validate

import (
	"context"
	"fmt"
	"reflect"
	"unicode/utf8"

	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"github.com/pkg/errors"
)

type maxLength int

func MaxLength(max int) Validator {
	v := maxLength(max)
	return &v
}

func (t maxLength) Validate(_ context.Context, field *field.Field, value interface{}) error {
	if s, ok := value.(string); ok {
		n := utf8.RuneCountInString(s)
		if t > 0 && n > int(t) {
			return fmt.Errorf("maximum string length is %d, got \"%s\" (length=%d)", t, s, len(s))
		}
		return nil
	}
	return fmt.Errorf("incorrect type: \"%s\", expected \"string\"", reflect.ValueOf(value).Kind())
}

type minLength int

func MinLength(max int) Validator {
	v := minLength(max)
	return &v
}

func (t minLength) Validate(_ context.Context, field *field.Field, value interface{}) error {
	if s, ok := value.(string); ok {
		n := utf8.RuneCountInString(s)
		if n < int(t) {
			return fmt.Errorf("minimum string length is %d, got \"%s\" (length=%d)", t, s, len(s))
		}
		return nil
	}
	return fmt.Errorf("incorrect type: \"%s\", expected \"string\"", reflect.ValueOf(value).Kind().String())
}

type schema bool

func Schema() Validator {
	v := schema(true)
	return &v
}

func (t schema) Validate(_ context.Context, _ *field.Field, value interface{}) error {
	if value == nil {
		return nil
	}
	if s, ok := value.(string); ok {
		if s == "" {
			return nil
		}
		sch := field.Object()
		if err := sch.UnmarshalJSON([]byte(s)); err != nil {
			return errors.New("value is not valid schema")
		}
		return nil
	}
	return errors.Errorf("incorrect type: \"%s\", expected string", reflect.ValueOf(value).Kind())
}
