package validate

import (
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"github.com/stretchr/testify/require"
)

func TestRequired(t *testing.T) {
	tests := []struct {
		name    string
		field   *field.Field
		data    interface{}
		wantErr bool
	}{
		{"Array no value", field.Array(field.String(), Required()), nil, true},
		{"Array empty", field.Array(field.String(), Required()), []interface{}{}, true},
		{"Array not empty", field.Array(field.String(), Required()), []interface{}{"a", "b"}, false},
		{"Boolean no value", field.Bool(Required()), nil, true},
		{"Boolean", field.Bool(Required()), false, false},
		{"Location: no value", field.Location(Required()), nil, true},
		//{"Location: empty", field.Location(Required()), map[string]interface{}{}, true}, // не имеет смысла, при Decode вернется ошибка если объект пустой
		{"Location: not empty", field.Location(Required()), &field.GeoObject{Address: "addr"}, false},
		{"Number (int) no value", field.Number(field.NumberFormatInt, Required()), nil, true},
		{"Number (int) empty", field.Number(field.NumberFormatInt, Required()), 0, false},
		{"Number (int) not empty", field.Number(field.NumberFormatInt, Required()), 42, false},
		{"Number (float) no value", field.Number(field.NumberFormatFloat, Required()), nil, true},
		{"Number (float) empty", field.Number(field.NumberFormatFloat, Required()), 0.0, false},
		{"Number (float) not empty", field.Number(field.NumberFormatFloat, Required()), 42.2, false},
		{"Object no value", field.Object("a", field.String(), "b", field.Bool(), Required()), nil, true},
		{"Object no value", field.Object("a", field.String(), "b", field.Bool(), Required()), map[string]interface{}{}, true},
		{"Object not empty", field.Object("a", field.String(), "b", field.Bool(), Required()), map[string]interface{}{"b": true}, false},
		{"String no value", field.String(Required()), nil, true},
		{"String empty", field.String(Required()), "", true},
		{"String not empty", field.String(Required()), "1234567", false},
		{"Time no value", field.Time(Required()), nil, true},
		{"Time not empty", field.Time(Required()), time.Now().Format(time.RFC3339), false},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := field.Decode(nil, tt.field, tt.data)
			require.NoError(t, err)
			err = Validate(nil, tt.field, got)
			if (err != nil) != tt.wantErr {
				t.Errorf("Validate() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
		})
	}
}
