package validate

import (
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestNumber(t *testing.T) {
	tests := []struct {
		name    string
		field   *field.Field
		data    interface{}
		wantErr bool
	}{
		{"Max Int", field.Number("int").AddOptions(Max(5.0)), 6, true},
		{"Min Int", field.Number("float").AddOptions(Min(10)), 9.0, true},

		{"Max Float", field.Number("float").AddOptions(Max(5.5)), 5.6, true},
		{"Min Float", field.Number("float").AddOptions(Min(10.1)), 9.9, true},

		{"MultipleOf Int", field.Number("float").AddOptions(MultipleOf(5)), 1.0, true},
		{"MultipleOf Int", field.Number("float").AddOptions(MultipleOf(5)), 10.0, false},

		{"MultipleOf Float", field.Number("float").AddOptions(MultipleOf(0.01)), 0.2, true},
		{"MultipleOf Float", field.Number("float").AddOptions(MultipleOf(0.05)), 0.1, false},

		{"MinMaxMultipleOf Int", field.Number("int").AddOptions(Max(5), Min(2), MultipleOf(5)), 6, true},
		{"MinMaxMultipleOf Int", field.Number("int").AddOptions(Max(5), Min(2), MultipleOf(2)), 4, false},
		{"MinMaxMultipleOf Int", field.Number("int").AddOptions(Max(5), Min(1), MultipleOf(2)), -2, true},
		{"MinMaxMultipleOf Int", field.Number("int").AddOptions(Max(5), Min(-1), MultipleOf(2)), 0, false},
		{"MinMaxMultipleOf Int", field.Number("int").AddOptions(Max(10), Min(7), MultipleOf(5)), 6, true},
		{"MinMaxMultipleOf Float", field.Number("int").AddOptions(Max(5.5), Min(2.3), MultipleOf(5)), 6, true},
		{"MinMaxMultipleOf Float", field.Number("float").AddOptions(Max(10.1), Min(7.7), MultipleOf(0.1)), 6.1, true},

		{"Enum miss", field.Number("int").AddOptions(Enum(EnumOpt{Name: "N 1", Value: 1}, EnumOpt{Name: "N 2", Value: 2})), 3, true},
		{"Enum match", field.Number("int").AddOptions(Enum(EnumOpt{Name: "N 1", Value: 1}, EnumOpt{Name: "N 2", Value: 2})), 2, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := field.Decode(nil, tt.field, tt.data)
			require.NoError(t, err)
			err = Validate(nil, tt.field, got)
			if (err != nil) != tt.wantErr {
				t.Errorf("Validate() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
		})
	}
}

func TestNumberValidate(t *testing.T) {
	tests := []struct {
		name    string
		field   *field.Field
		data    interface{}
		wantErr bool
		error   string
	}{
		{"Nil Max", field.Number("int").AddOptions(Max(1.0)), nil, false, ""},
		{"Nil Min", field.Number("int").AddOptions(Min(1.0)), nil, false, ""},
		{"Nil MultipleOf", field.Number("int").AddOptions(MultipleOf(1)), nil, false, ""},
		{"Int Max", field.Number("int").AddOptions(Max(1.0)), 1, false, ""},
		{"Int Min", field.Number("int").AddOptions(Min(1.0)), 1, false, ""},
		{"Int MultipleOf", field.Number("int").AddOptions(Min(1.0)), 1, false, ""},
		{"Flоat Max", field.Number("float").AddOptions(Max(1.0)), 1.0, false, ""},
		{"Flоat Min", field.Number("float").AddOptions(Min(1.0)), 1.0, false, ""},
		{"Flоat MultipleOf", field.Number("float").AddOptions(Min(1.0)), 1.0, false, ""},
		{"Bool Max Num", field.Number("int").AddOptions(Max(1.0)), true, true, "validation error: incorrect type: \"bool\", expected \"number\""},
		{"Bool Min Num", field.Number("int").AddOptions(Min(1.0)), true, true, "validation error: incorrect type: \"bool\", expected \"number\""},
		{"Bool MultipleOf Num", field.Number("int").AddOptions(MultipleOf(1)), true, true, "validation error: incorrect type: \"bool\", expected \"number\""},
		{"String Max Num", field.Number("int").AddOptions(Max(1.0)), "1", true, "validation error: incorrect type: \"string\", expected \"number\""},
		{"String Min Num", field.Number("int").AddOptions(Min(1.0)), "1", true, "validation error: incorrect type: \"string\", expected \"number\""},
		{"String MultipleOf Num", field.Number("int").AddOptions(MultipleOf(1)), "1", true, "validation error: incorrect type: \"string\", expected \"number\""},
		{"Array Max Num", field.Number("int").AddOptions(Max(1.0)), [1]int{1}, true, "validation error: incorrect type: \"array\", expected \"number\""},
		{"Array Min Num", field.Number("int").AddOptions(Min(1.0)), [1]int{1}, true, "validation error: incorrect type: \"array\", expected \"number\""},
		{"Slice Max Num", field.Number("int").AddOptions(Max(1.0)), []int{1}, true, "validation error: incorrect type: \"slice\", expected \"number\""},
		{"Slice Min Num", field.Number("int").AddOptions(Min(1.0)), []int{1}, true, "validation error: incorrect type: \"slice\", expected \"number\""},
		{"Slice MultipleOf Num", field.Number("int").AddOptions(MultipleOf(1)), []int{}, true, "validation error: incorrect type: \"slice\", expected \"number\""},
		{"Map Max Num", field.Number("int").AddOptions(Max(1.0)), map[int]int{}, true, "validation error: incorrect type: \"map\", expected \"number\""},
		{"Map Min Num", field.Number("int").AddOptions(Min(1.0)), map[int]int{}, true, "validation error: incorrect type: \"map\", expected \"number\""},
		{"Map MultipleOf Num", field.Number("int").AddOptions(MultipleOf(1)), map[int]int{}, true, "validation error: incorrect type: \"map\", expected \"number\""},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			err := Validate(nil, tt.field, tt.data)
			if tt.wantErr {
				require.Error(t, err)
				assert.EqualError(t, err, tt.error)
			}
			if !tt.wantErr {
				require.NoError(t, err)
			}
		})
	}
}
