package test

import (
	"context"
	"encoding/json"
	"fmt"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/data"
	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"git.perx.ru/perxis/perxis-go/pkg/schema/modify"
	"git.perx.ru/perxis/perxis-go/pkg/schema/validate"
	"github.com/hashicorp/go-multierror"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestDefaultTimeField_JSON(t *testing.T) {
	w, _ := time.Parse(time.RFC3339, "2012-11-01T22:08:41Z")
	fld := field.Object(
		"timeField", field.Time(modify.Default(w)),
	)

	b, err := json.MarshalIndent(fld, "", "  ")
	require.NoError(t, err)

	res := &field.Field{}
	err = json.Unmarshal(b, res)
	require.NoError(t, err)

	assert.Equal(t, fld, res)
}

func TestStringField_JSON(t *testing.T) {
	fld := field.String().AddOptions(validate.MaxLength(200), modify.TrimSpace())

	b, err := json.MarshalIndent(fld, "", "  ")
	require.NoError(t, err)

	res := field.NewField(nil)
	err = json.Unmarshal(b, res)
	require.NoError(t, err)

	assert.Equal(t, fld, res)
}

func TestNumberField_JSON(t *testing.T) {
	fld := field.Number(field.NumberFormatInt).AddOptions(
		validate.Min(0),
		validate.Max(10),
		validate.MultipleOf(2),
		validate.Enum(
			validate.EnumOpt{
				Name:  "N 1",
				Value: 1.0,
			},
			validate.EnumOpt{
				Name:  "N 2",
				Value: 2.0,
			},
		),
	)

	b, err := json.MarshalIndent(fld, "", "  ")
	require.NoError(t, err)
	//fmt.Println(string(b))

	res := field.NewField(nil)
	err = json.Unmarshal(b, res)
	require.NoError(t, err)

	assert.Equal(t, fld, res)
}

func TestSchema_JSON(t *testing.T) {
	enumStr := field.String().AddOptions(
		validate.Enum(
			validate.EnumOpt{
				Name:  "N 1",
				Value: "n1",
			}, validate.EnumOpt{
				Name:  "N 2",
				Value: "n2",
			},
		),
	).SetAdditionalValues()
	enumInt := field.Number(field.NumberFormatFloat).AddOptions(
		validate.Enum(
			validate.EnumOpt{
				Name:  "N 1",
				Value: 1.1,
			}, validate.EnumOpt{
				Name:  "N 2",
				Value: 2.5,
			},
		),
	)
	sch := schema.New(
		"stringField", field.String().WithUI(&field.UI{Placeholder: "Test name"}).AddOptions(modify.TrimSpace()).AddOptions(validate.MinLength(2), validate.MaxLength(10)),
		"stringField2", field.String(modify.Default("default")),
		"intField", field.Number("int", validate.Required()),
		"floatField", field.Number("float").SetIndexed(true),
		"enumStringField", enumStr,
		"enumIntField", enumInt,
		"timeField", field.Time().SetSingleLocale(true),
		"arrayField", field.Array(field.String(modify.Default("default"))),
		"objectField", field.Object("innerStringField", field.String()).WithIncludes("ref1", field.Include{Ref: "ref2", Optional: true}),
		"evaluatedField", field.String(modify.Value("stringField2 + '_' 	")),
	)
	sch.Loaded = true

	b, err := json.MarshalIndent(sch, "", "  ")
	require.NoError(t, err)
	//fmt.Println(string(b))

	res := schema.New()
	err = json.Unmarshal(b, res)
	require.NoError(t, err)

	assert.Equal(t, sch, res)
}

func TestSchemaUI_UnmarshalJSON(t *testing.T) {
	vw := &field.View{
		Widget:  "Widget",
		Options: map[string]interface{}{"title": "name", "key": "name"},
	}
	ui := &field.UI{
		Widget:      "Widget",
		Placeholder: "Placeholder",
		Options:     map[string]interface{}{"title": "name", "key": "name"},
		ListView:    vw,
		ReadView:    vw,
		EditView:    vw,
	}
	schm := schema.New(
		"name", field.String().WithUI(ui),
	)
	schm.UI = ui

	j := `{
  "ui": {
    "widget": "Widget",
    "placeholder": "Placeholder",
    "options": {
      "title": "name",
      "key": "name"
    },
    "read_view": {
      "widget": "Widget",
      "options": {
        "title": "name",
        "key": "name"
      }
    },
    "edit_view": {
      "widget": "Widget",
      "options": {
        "title": "name",
        "key": "name"
      }
    },
    "list_view": {
      "widget": "Widget",
      "options": {
        "title": "name",
        "key": "name"
      }
    }
  },
  "type": "object",
  "params": {
    "inline": false,
    "fields": {
      "name": {
        "ui": {
          "widget": "Widget",
          "placeholder": "Placeholder",
          "options": {
            "title": "name",
            "key": "name"
          },
          "read_view": {
            "widget": "Widget",
            "options": {
              "title": "name",
              "key": "name"
            }
          },
          "edit_view": {
            "widget": "Widget",
            "options": {
              "title": "name",
              "key": "name"
            }
          },
          "list_view": {
            "widget": "Widget",
            "options": {
              "title": "name",
              "key": "name"
            }
          }
        },
        "type": "string",
        "params": {}
      }
    }
  },
  "loaded": false
}`

	sch := schema.New()
	err := sch.UnmarshalJSON([]byte(j))
	require.NoError(t, err)
	assert.Equal(t, sch, schm)
}

func TestSchema_GetField(t *testing.T) {

	sch := schema.New(
		"str", field.String(),
		"num", field.Number(field.NumberFormatInt),
		"obj", field.Object(
			"bool", field.Bool(),
			"arr", field.Array(field.Time()),
			"list", field.Array(
				field.Object(
					"num1", field.Number(field.NumberFormatFloat),
					"str1", field.String(),
					"obj1", field.Object(
						"str2", field.String(),
					),
				),
			),
			"geo", field.Location(),
		),
	)

	data := []struct {
		fld    string
		exists bool
		typ    string
	}{
		{"str", true, "string"},
		{"obj.bool", true, "bool"},
		{"obj.list.num1", true, "number"},
		{"obj.list.obj1.str2", true, "string"},
		{"obj_list", false, ""},
		{"zzz", false, ""},
		{"obj.geo", true, "location"},
	}

	for _, d := range data {
		t.Run(d.fld, func(t *testing.T) {
			f := sch.GetField(d.fld)
			if d.exists {
				require.NotNil(t, f, fmt.Sprintf("not found '%s'", d.fld))
				assert.Equal(t, d.typ, f.GetType().Name(), fmt.Sprintf("field '%s'", d.fld))
			} else {
				require.Nil(t, f)
			}
		})
	}
}

func TestSchema_GetField_WithInline(t *testing.T) {

	sch := schema.New(
		"str", field.String(),
		"obj1", field.Object(
			true,
			"obj11", field.Object(
				true,
				"obj111", field.Object(
					true,
					"str1", field.String(),
					"str2", field.String(),
				),
				"arr1", field.Array(field.Object(
					"str3", field.String(),
				)),
				"arr2", field.Array(field.String()),
			),
		),
		"obj2", field.Object(
			true,
			"a", field.String(),
			"b", field.String(),
		),
		"zz", field.Object(
			true,
			"zz", field.Array(field.Object(
				"str3", field.String(),
			)),
		),
	)

	data := []struct {
		fld    string
		exists bool
		typ    string
	}{
		{"str", true, "string"},
		{"a", true, "string"},
		{"b", true, "string"},
		{"str1", true, "string"},
		{"str2", true, "string"},
		{"arr1", true, "array"},
		{"arr2", true, "array"},
		{"arr1.str3", true, "string"},
		{"zz.str3", true, "string"},
	}

	for _, d := range data {
		t.Run(d.fld, func(t *testing.T) {
			f := sch.GetField(d.fld)
			if d.exists {
				require.NotNil(t, f, fmt.Sprintf("not found '%s'", d.fld))
				assert.Equal(t, d.typ, f.GetType().Name(), fmt.Sprintf("field '%s'", d.fld))
			} else {
				require.Nil(t, f)
			}
		})
	}
}

func TestSchema_GetFields(t *testing.T) {
	sch := schema.New(
		"str", field.String().SetTitle("Str"),
		"num", field.Number(field.NumberFormatInt).SetIndexed(true).SetTitle("Num"),
		"obj", field.Object(
			"arr", field.Array(
				field.Array(field.Time()).SetIndexed(true).SetTitle("NestedArr"),
			).SetTitle("Arr"),
			"list", field.Array(
				field.Object(
					"obj3", field.Object(
						"str", field.String().SetIndexed(true).SetTitle("Obj2.List.Str"),
					).SetTitle("Obj3"),
				).SetTitle("Obj2"),
			).SetTitle("List"),
			"geo", field.Location().SetTitle("Geo"),
		).SetTitle("Obj"),
	)

	flds := sch.GetFields(func(f *field.Field, path string) bool { return true })
	assert.Len(t, flds, 8)

	paths := make([]string, 0, len(flds))

	for _, fld := range flds {
		switch fld.Path {
		case "str":
			assert.IsType(t, &field.StringParameters{}, fld.Params)
		case "num":
			assert.IsType(t, &field.NumberParameters{}, fld.Params)
		case "obj":
			assert.IsType(t, &field.ObjectParameters{}, fld.Params)
		case "obj.arr":
			assert.IsType(t, &field.ArrayParameters{}, fld.Params)
		case "obj.list":
			assert.IsType(t, &field.ArrayParameters{}, fld.Params)
		case "obj.list.obj3":
			assert.IsType(t, &field.ObjectParameters{}, fld.Params)
		case "obj.list.obj3.str":
			assert.IsType(t, &field.StringParameters{}, fld.Params)
		case "obj.geo":
			assert.IsType(t, &field.LocationParameters{}, fld.Params)

		}

		paths = append(paths, fld.Path)
	}

	assert.ElementsMatch(
		t,
		[]string{"str", "num", "obj", "obj.arr", "obj.list", "obj.list.obj3", "obj.list.obj3.str", "obj.geo"},
		paths,
	)

}

func TestSchema_GetFieldByPath(t *testing.T) {
	sch := schema.New(
		"str", field.String().SetTitle("Str"),
		"num", field.Number(field.NumberFormatInt).SetIndexed(true).SetTitle("Num"),
		"obj", field.Object(
			"arr", field.Array(
				field.Array(field.Time()).SetIndexed(true).SetTitle("NestedArr"),
			).SetTitle("Arr"),
			"list", field.Array(
				field.Object(
					"obj3", field.Object(
						"str", field.String().SetIndexed(true).SetTitle("Obj2.List.Str"),
					).SetTitle("Obj3"),
				).SetTitle("Obj2"),
			).SetTitle("List"),
			"geo", field.Location().SetTitle("Geo"),
		).SetTitle("Obj"),
	)

	dt := []struct {
		name  string
		paths []string
		want  []string
	}{
		{
			"all",
			[]string{"*"},
			[]string{"str", "num", "obj", "obj.arr", "obj.list", "obj.list.obj3", "obj.list.obj3.str", "obj.geo"},
		},
		{
			"full match",
			[]string{"str", "obj.list.obj3", "some"},
			[]string{"str", "obj.list.obj3"},
		},
		{
			"glob",
			[]string{"str*", "obj.list*", "*geo"},
			[]string{"str", "obj.list", "obj.list.obj3", "obj.list.obj3.str", "obj.geo"},
		},
	}

	for _, d := range dt {
		t.Run(d.name, func(t *testing.T) {
			got := field.GetFieldsPath(sch.GetFields(func(f *field.Field, path string) bool {
				return data.GlobMatch(path, d.paths...)
			}))
			assert.ElementsMatch(t, d.want, got)
		})
	}
}

func TestSchema_GetFieldsInline(t *testing.T) {
	t.Run("Basic", func(t *testing.T) {
		sch := schema.New(
			"str_1", field.String(),
			"num", field.Number(field.NumberFormatInt).SetIndexed(true),
			"obj_1", field.Object(
				"arr", field.Array(field.Time()).SetIndexed(true),
				"list", field.Array(
					field.Object(
						"obj_2", field.Object(
							"str_2", field.String().SetIndexed(true),
						),
					),
				),
				"geo", field.Location(),
			),
		)

		flds := field.GetFieldsPath(sch.GetFields(func(f *field.Field, p string) bool {
			return true
		}, "data"))
		assert.ElementsMatch(
			t,
			[]string{
				"data",
				"data.str_1",
				"data.num",
				"data.obj_1",
				"data.obj_1.arr",
				"data.obj_1.list",
				"data.obj_1.list.obj_2",
				"data.obj_1.list.obj_2.str_2",
				"data.obj_1.geo",
			},
			flds,
		)
	})
	t.Run("Inline fields in schema in a row", func(t *testing.T) {
		sch := schema.New(
			"obj_inline_1", field.Object(
				true,
				"inline_field1", field.String().SetUnique(true),
				"obj_inline_2", field.Object(true,
					"inline_field2", field.String(),
					"arr", field.Array(field.Object(true,
						"inline_field3", field.String(),
					)),
				),
			),
		)

		flds := field.GetFieldsPath(sch.GetFields(func(f *field.Field, p string) bool {
			return true
		}))
		assert.ElementsMatch(
			t,
			[]string{
				"obj_inline_1",
				"inline_field1",
				"obj_inline_2",
				"inline_field2",
				"arr",
				"arr.inline_field3",
			},
			flds,
		)
	})
	t.Run("Inline fields in schema in a row with prefix", func(t *testing.T) {
		sch := schema.New(
			"obj_inline_1", field.Object(true,
				"inline_field1", field.String().SetUnique(true),
				"obj_inline_2", field.Object(true,
					"inline_field2", field.String(),
					"obj_inline_3", field.Object(true,
						"inline_field3", field.String(),
					),
				),
			),
		)

		flds := field.GetFieldsPath(sch.GetFields(func(f *field.Field, p string) bool {
			return true
		}, "data"))
		assert.ElementsMatch(
			t,
			[]string{
				"data",
				"data.obj_inline_1",
				"data.inline_field1",
				"data.obj_inline_2",
				"data.inline_field2",
				"data.obj_inline_3",
				"data.inline_field3",
			},
			flds,
		)
	})
	t.Run("Mixed fields in schema in a row", func(t *testing.T) {
		sch := schema.New(
			"obj_not_inline_1", field.Object(
				"not_inline_field_1", field.String().SetUnique(true),
				"obj_inline_1", field.Object(true,
					"inline_field1", field.String(),
					"obj_not_inline_2", field.Object(
						"not_inline_field_2", field.String(),
						"obj_inline_2", field.Object(true,
							"inline_field2", field.String(),
						),
					),
				),
			),
		)

		flds := field.GetFieldsPath(sch.GetFields(func(f *field.Field, p string) bool {
			return true
		}, "data"))
		assert.ElementsMatch(
			t,
			[]string{
				"data",
				"data.obj_not_inline_1",
				"data.obj_not_inline_1.not_inline_field_1",
				"data.obj_not_inline_1.obj_inline_1",
				"data.obj_not_inline_1.inline_field1",
				"data.obj_not_inline_1.obj_not_inline_2",
				"data.obj_not_inline_1.obj_not_inline_2.not_inline_field_2",
				"data.obj_not_inline_1.obj_not_inline_2.obj_inline_2",
				"data.obj_not_inline_1.obj_not_inline_2.inline_field2",
			},
			flds,
		)
	})
}

func TestSchema_Clone(t *testing.T) {
	sch := schema.New(
		"f", field.String().WithUI(&field.UI{Placeholder: "Test name"}).AddOptions(modify.TrimSpace()).AddTranslation("ru", "ф", "Поле Ф"),
		"obj", field.Object(
			"list", field.Array(
				field.Object(
					"obj", field.Object(
						"field", field.String(),
					),
				),
			),
		),
	)

	t.Run("Simple", func(t *testing.T) {
		f := sch.GetField("f")
		fld := f.Clone(false)

		assert.Equal(t, f.UI, fld.UI)
		assert.Equal(t, f.Options, fld.Options)
		assert.Equal(t, f.Translations, fld.Translations)
		assert.Equal(t, f.Params, fld.Params)
	})

	t.Run("Reset", func(t *testing.T) {
		f := sch.GetField("obj")
		fld := f.Clone(true)

		assert.Equal(t, f.UI, fld.UI)
		assert.Equal(t, f.Options, fld.Options)
		assert.Equal(t, f.Translations, fld.Translations)
		assert.NotEqual(t, f.Params, fld.Params)

		f = sch.GetField("obj.list")
		fld = f.Clone(true)

		assert.Equal(t, f.UI, fld.UI)
		assert.Equal(t, f.Options, fld.Options)
		assert.Equal(t, f.Translations, fld.Translations)
		assert.NotEqual(t, f.Params, fld.Params)

		f = sch.GetField("obj.list.obj")
		fld = f.Clone(true)

		assert.Equal(t, f.UI, fld.UI)
		assert.Equal(t, f.Options, fld.Options)
		assert.Equal(t, f.Translations, fld.Translations)
		assert.NotEqual(t, f.Params, fld.Params)
	})
}

func TestSchema_Modify(t *testing.T) {
	sch := schema.New(
		"name", field.String(validate.Required()),
		"last_name", field.String(validate.Required()),
		"got_nobel", field.Bool(),
		"times", field.Number("int"),
		"dates", field.Array(field.Time()),
	)

	in := map[string]interface{}{"last_name": "Curie", "name": "Marie"}
	_, _, err := modify.Modify(nil, sch, in)
	require.NoError(t, err)
}

func TestSchema_Validate(t *testing.T) {
	sch := schema.New(
		"name", field.String(validate.Required()),
		"last_name", field.String(),
		"info", field.Object(
			"time", field.Time(),
			"numbers", field.Number(
				field.NumberFormatInt,
				validate.Enum(
					validate.EnumOpt{Name: "first", Value: 1},
					validate.EnumOpt{Name: "second", Value: 2},
				),
			),
		),
	)

	in := map[string]interface{}{"info": map[string]interface{}{"time": time.Now()}, "name": "Name"}
	err := validate.Validate(nil, sch, in)
	require.NoError(t, err)
}

func TestSchema_DecodeErrors(t *testing.T) {
	sch := schema.New(
		"name", field.String(validate.Required()),
		"last_name", field.String(),
		"a", field.Object(
			"time", field.Time(),
			"num1", field.Number(field.NumberFormatInt),
			"num2", field.Number(field.NumberFormatInt),
			"num3", field.Number(field.NumberFormatInt),
			"b", field.Object(
				"num1", field.Number(field.NumberFormatInt),
				"num2", field.Number(field.NumberFormatInt),
				"num3", field.Number(field.NumberFormatInt),
			),
			"c", field.Array(field.Number(field.NumberFormatInt)),
			"d", field.Number(field.NumberFormatInt, validate.Max(10)),
		),
	)

	in := map[string]interface{}{"a": map[string]interface{}{"time": time.Now(), "num1": "a", "num2": "b", "num3": "c", "d": 20,
		"b": map[string]interface{}{"time": time.Now(), "num1": "a", "num2": "b", "num3": "c", "str": "s"}, "c": []interface{}{"a", "b", "c"}},
		"name": "Name"}
	_, err := schema.Decode(nil, sch, in)
	assert.Error(t, err)
	assert.Contains(t, err.Error(), "decode error")
}

func TestSchema_ValidateErrors(t *testing.T) {
	sch := schema.New(
		"a", field.Object(
			"num1", field.Number(field.NumberFormatInt, validate.Required()),
			"num2", field.Number(field.NumberFormatInt, validate.Max(10)),
			"num3", field.Number(field.NumberFormatInt, validate.Min(10)),
			"str1", field.String(validate.MaxLength(5)),
			"str2", field.String(validate.MinLength(5)),
			"str3", field.String(validate.MinLength(5), validate.Enum(validate.EnumOpt{Value: "somesome"}, validate.EnumOpt{Value: "romoromo"})),
		),
	)

	in := map[string]interface{}{"a": map[string]interface{}{"num2": 20, "num3": 5, "str1": "123456", "str2": "123", "str3": "some"}}
	decoded, err := schema.Decode(nil, sch, in)
	require.NoError(t, err)
	err = validate.Validate(nil, sch, decoded)
	require.Error(t, err)
	require.Contains(t, err.Error(), "validation error")
	var merr *multierror.Error
	require.ErrorAs(t, err, &merr)
	assert.Len(t, merr.Errors, 6)
}

func TestSchema_ValidateEmptyObject(t *testing.T) {
	{
		sch := schema.New(
			"num1", field.Number(field.NumberFormatInt, validate.Required()),
		)

		res, err := schema.Decode(nil, sch, nil)
		require.NoError(t, err)
		res, _, err = modify.Modify(nil, sch, res)
		require.NoError(t, err)
		err = validate.Validate(nil, sch, res)
		require.NoError(t, err, "поля объекта nil не проверяются")
	}
	{
		sch := schema.New(
			"num1", field.Number(field.NumberFormatInt, validate.Required()),
		)

		res, err := schema.Decode(nil, sch, map[string]interface{}{})
		require.NoError(t, err)
		res, _, err = modify.Modify(nil, sch, res)
		require.NoError(t, err)
		err = validate.Validate(nil, sch, res)
		require.Error(t, err, "поля пустого объекта проверяются")
	}
	{
		sch := schema.New(
			"num1", field.Number(field.NumberFormatInt, validate.Required()),
		)

		res, err := schema.Decode(nil, sch, map[string]interface{}{"a": "sss"})
		require.NoError(t, err)
		res, _, err = modify.Modify(nil, sch, res)
		require.NoError(t, err)
		err = validate.Validate(nil, sch, res)
		require.Error(t, err, "поля объекта с некорректными данными проверяются")
	}

	{
		sch := schema.New(
			"num1", field.Number(field.NumberFormatInt, validate.Required()),
		).AddOptions(modify.Default(map[string]interface{}{}))

		res, err := schema.Decode(nil, sch, nil)
		require.NoError(t, err)
		res, _, err = modify.Modify(nil, sch, res)
		require.NoError(t, err)
		err = validate.Validate(nil, sch, res)
		require.Error(t, err, "поля nil объекта Default данными проверяются")
	}
}

func TestSchema_ModificationErrors(t *testing.T) {
	sch := schema.New(
		"a", field.Object(
			"num1", field.Number(field.NumberFormatInt, modify.TrimSpace()),
			"str1", field.String(modify.TrimSpace()),
		),
	)

	in := map[string]interface{}{"a": map[string]interface{}{"num1": 20, "num3": 5, "str1": "123456", "str2": "123", "str3": "some"}}
	decoded, err := schema.Decode(nil, sch, in)
	require.NoError(t, err)
	_, _, err = modify.Modify(nil, sch, decoded)
	require.Error(t, err)
	require.Contains(t, err.Error(), "modification error")
	var merr *multierror.Error
	require.ErrorAs(t, err, &merr)
	assert.Len(t, merr.Errors, 1)
}

func TestSchema_UnknownJSON(t *testing.T) {
	sch := schema.New(
		"name", field.String(validate.Required()),
		"last_name", field.String(validate.Required()),
		"got_nobel", field.Bool(),
		"times", field.Number("int"),
		"dates", field.Array(field.Time()),
	)

	b, err := json.Marshal(sch)
	require.NoError(t, err)
	field.Unregister("object")

	s1 := schema.New()
	err = json.Unmarshal(b, s1)
	require.NoError(t, err)
	assert.Equal(t, "unknown", s1.GetType().Name(), "Схема неизвестного типа должна определяться как unknown")

	in := map[string]interface{}{"info": map[string]interface{}{"time": time.Now()}, "name": "Name"}
	out, err := field.Decode(nil, s1, in)
	require.NoError(t, err)
	assert.Equal(t, in, out, "Данные неизвестного типа не изменяются при декодировании")
	err = validate.Validate(nil, s1, in)
	require.NoError(t, err, "Данные неизвестного типа не валидируются вглубь")

	b, err = json.Marshal(s1)
	require.NoError(t, err)
	s2 := schema.New()
	err = json.Unmarshal(b, s2)
	require.NoError(t, err)
	b, err = json.Marshal(s2)
	require.NoError(t, err)
	assert.Equal(t, "unknown", s2.GetType().Name(), "Схема неизвестного типа должна определяться как unknown")
	assert.Equal(t, s1, s2, "Схема не должна меняться при повторном маршалинге")

	field.Register(&field.ObjectType{})
	s3 := schema.New()
	err = json.Unmarshal(b, s3)
	require.NoError(t, err)
	assert.Equal(t, "object", s3.GetType().Name(), "Схема должна восстановить тип object при восстановление регистрации типа")
	assert.Equal(t, sch, s3, "Схема должна восстановиться при восстановление регистрации типа")
}

func TestSchema_ValidOptions(t *testing.T) {
	t.Run("Valid Options", func(t *testing.T) {
		schm := `{
	"type": "object",
	"params": {
		"fields": {
			"required": {
				"options": {
					"required": true
				},
				"type": "string"
			},
			"readonly": {
				"options": {
					"readonly": true
				},
				"type": "string"
			},
			"enum": {
				"options": {
					"enum": [{
							"name": "One",
							"value": "one"
						},
						{
							"name": "Two",
							"value": "two"
						}
					]
				},
				"type": "string"
			}
		}
	}
}`

		s := schema.New()
		err := json.Unmarshal([]byte(schm), s)
		require.NoError(t, err)

		required := s.GetField("required")
		readonly := s.GetField("readonly")
		enum := s.GetField("enum")

		require.NotEmpty(t, required.Options)
		require.NotEmpty(t, readonly.Options)
		require.NotEmpty(t, enum.Options)
	})

	t.Run("Invalid Options", func(t *testing.T) {
		schm := `{
	"type": "object",
	"params": {
		"fields": {
			"required": {
				"options": {
					"required": false
				},
				"type": "string"
			},
			"readonly": {
				"options": {
					"readonly": false
				},
				"type": "string"
			}
		}
	}
}`

		s := schema.New()
		err := json.Unmarshal([]byte(schm), s)
		require.NoError(t, err)

		required := s.GetField("required")
		readonly := s.GetField("readonly")

		require.Empty(t, required.Options)
		require.Empty(t, readonly.Options)
	})

	t.Run("Required Enum Name", func(t *testing.T) {
		schm := `{
	"type": "object",
	"params": {
		"fields": {
			"enum": {
				"options": {
					"enum": [{
							"value": "one"
						},
						{
							"value": "two"
						}
					]
				},
				"type": "string"
			}
		}
	}
}`
		s := schema.New()
		err := json.Unmarshal([]byte(schm), s)
		require.Error(t, err)
		assert.Contains(t, err.Error(), "enum name is required")
	})
}

func TestSchema_Condition(t *testing.T) {
	sch := schema.New(
		"type", field.String(modify.TrimSpace()),
		"a", field.Number(field.NumberFormatInt).SetCondition("type contains 'a'"),
		"b", field.Number(field.NumberFormatInt, validate.Required()).SetCondition("type contains 'b'"),
		"c", field.Number(field.NumberFormatInt).SetCondition("a==10"),
		"obj", field.Object(
			"a", field.Number(field.NumberFormatInt).SetCondition("type contains 'a'"),
			"b", field.Number(field.NumberFormatInt).SetCondition("type contains 'b'"),
			"c", field.Number(field.NumberFormatInt).SetCondition("_.a < 10"),
			"d", field.Number(field.NumberFormatInt, modify.Default(11)).SetCondition("_.a < 10"),
		),
		"obj3", field.Object(
			"fld1", field.Number(field.NumberFormatInt),
		).SetCondition("obj.d > 10"),
	)

	tests := []struct {
		name    string
		data    map[string]interface{}
		want    map[string]interface{}
		wantErr bool
	}{
		{"type a",
			map[string]interface{}{
				"type": "a",
				"a":    int64(10),
				"b":    int64(10),
				"c":    int64(1),
				"obj":  map[string]interface{}{"a": int64(1), "b": int64(20), "c": int64(11), "d": int64(11)},
				"obj3": map[string]interface{}{"fld1": int64(6)},
			},
			map[string]interface{}{
				"type": "a",
				"a":    int64(10),
				"c":    int64(1),
				"obj":  map[string]interface{}{"a": int64(1), "c": int64(11), "d": int64(11)},
				"obj3": map[string]interface{}{"fld1": int64(6)},
			},
			false},
		{"type b",
			map[string]interface{}{
				"type": "b",
				"a":    int64(10),
				"b":    int64(10),
				"c":    int64(1),
				"obj":  map[string]interface{}{"a": int64(1), "b": int64(20), "c": int64(11), "d": int64(11)},
				"obj3": map[string]interface{}{"fld1": int64(6)},
			},
			map[string]interface{}{
				"type": "b",
				"b":    int64(10),
				"obj":  map[string]interface{}{"b": int64(20)},
			},
			false},
		{"type ab + default",
			map[string]interface{}{
				"type": " ab  ",
				"a":    int64(1),
				"b":    int64(10),
				"c":    int64(1),
				"obj":  map[string]interface{}{"a": int64(1), "b": int64(20), "c": int64(11)},
				"obj3": map[string]interface{}{"fld1": int64(6)},
			},
			map[string]interface{}{
				"type": "ab",
				"a":    int64(1),
				"b":    int64(10),
				"obj":  map[string]interface{}{"a": int64(1), "b": int64(20), "c": int64(11), "d": int64(11)},
				"obj3": map[string]interface{}{"fld1": int64(6)},
			},
			false},
	}

	ctx := context.Background()
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := sch.ToValue(ctx, tt.data)
			if tt.wantErr {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
			assert.Equal(t, tt.want, got)
		})
	}

}

func TestSchema_Inline(t *testing.T) {
	sch := schema.New(
		"a", field.String(),
		"b", field.String(),
		"obj", field.Object(
			true,
			"c", field.String(),
			"d", field.Number(field.NumberFormatInt),
			"inner_obj", field.Object(
				true,
				"f", field.String(),
			),
		).SetCondition("a == 'universe'"),
		"overlap", field.Object(
			"obj1", field.Object(
				true,
				"f1", field.Number(field.NumberFormatInt),
				"f2", field.String(),
			),
			"obj2", field.Object(
				true,
				"f1", field.Number(field.NumberFormatInt),
				"f2", field.Number(field.NumberFormatInt),
			),
		),
		"arr", field.Array(
			field.Object(
				true,
				"x", field.String(),
			),
		),
	)

	tests := []struct {
		name    string
		data    map[string]interface{}
		want    map[string]interface{}
		wantErr bool
	}{
		{"Condition success",
			map[string]interface{}{"a": "universe", "b": "life", "c": "everything", "d": int64(42)},
			map[string]interface{}{"a": "universe", "b": "life", "c": "everything", "d": int64(42)},
			false,
		},
		{"Condition fail",
			map[string]interface{}{"a": "life", "b": "universe", "c": "everything", "d": int64(42)},
			map[string]interface{}{"a": "life", "b": "universe"},
			false,
		},
		{"Condition success, level 2 inline",
			map[string]interface{}{"a": "universe", "b": "life", "c": "everything", "d": int64(42), "f": "some"},
			map[string]interface{}{"a": "universe", "b": "life", "c": "everything", "d": int64(42), "f": "some"},
			false,
		},
		{"Condition fail, level 2 inline",
			map[string]interface{}{"a": "life", "b": "universe", "c": "everything", "d": int64(42), "f": "some"},
			map[string]interface{}{"a": "life", "b": "universe"},
			false,
		},
		{"Overlapped",
			map[string]interface{}{"overlap": map[string]interface{}{"f1": 42}},
			map[string]interface{}{"overlap": map[string]interface{}{"f1": int64(42)}},
			false,
		},
		{"Overlapped, type conflict",
			map[string]interface{}{"overlap": map[string]interface{}{"f1": 42, "f2": "everything"}},
			nil,
			true,
		},
		{"Array, ignore inline",
			map[string]interface{}{"a": "life", "b": "universe", "c": "everything", "d": int64(42), "x": "some", "arr": []interface{}{map[string]interface{}{"x": "some"}}},
			map[string]interface{}{"a": "life", "b": "universe", "arr": []interface{}{map[string]interface{}{"x": "some"}}},
			false,
		},
	}

	ctx := context.Background()
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := sch.ToValue(ctx, tt.data)
			if tt.wantErr {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
			assert.Equal(t, tt.want, got)
		})
	}
}

func TestSchema_Introspect(t *testing.T) {
	tests := []struct {
		name           string
		data           map[string]interface{}
		schema         *schema.Schema
		want           map[string]interface{}
		wantFields     []string
		dontWantFields []string
		wantErr        bool
	}{
		{"single true condition",
			map[string]interface{}{
				"a": "b",
				"b": "b",
			},
			schema.New(
				"a", field.String(),
				"b", field.String().SetCondition("a == 'b'"),
			),
			map[string]interface{}{
				"a": "b",
				"b": "b",
			},
			[]string{"a", "b"},
			[]string{},
			false},
		{"single false condition",
			map[string]interface{}{
				"a": "a",
				"b": "b",
			},
			schema.New(
				"a", field.String(),
				"b", field.String().SetCondition("a == 'b'"),
			),
			map[string]interface{}{
				"a": "a",
			},
			[]string{"a"},
			[]string{"b"},
			false},
		{"multiple true conditions",
			map[string]interface{}{
				"a": "a",
				"b": "b",
				"c": "c",
				"d": "d",
			},
			schema.New(
				"a", field.String(),
				"b", field.String().SetCondition("a == 'a'"),
				"c", field.String().SetCondition("b == 'b'"),
				"d", field.String().SetCondition("c == 'c'"),
			),
			map[string]interface{}{
				"a": "a",
				"b": "b",
				"c": "c",
				"d": "d",
			},
			[]string{"a", "b", "c", "d"},
			[]string{},
			false},
		{"multiple conditions some true",
			map[string]interface{}{
				"a": "a",
				"b": "bb",
				"c": "c",
				"d": "d",
			},
			schema.New(
				"a", field.String(),
				"b", field.String().SetCondition("a == 'a'"),
				"c", field.String().SetCondition("b == 'b'"),
				"d", field.String().SetCondition("c == 'c'"),
			),
			map[string]interface{}{
				"a": "a",
				"b": "bb",
			},
			[]string{"a", "b"},
			[]string{"c", "d"},
			false},
		{"nil data",
			nil,
			schema.New(
				"a", field.String(),
				"b", field.String(),
			),
			nil,
			[]string{"a", "b"},
			nil,
			false},
		{"empty data",
			map[string]interface{}{},
			schema.New(
				"a", field.String(),
				"b", field.String(),
			),
			map[string]interface{}{},
			[]string{"a", "b"},
			nil,
			false},
		{"data with other fields",
			map[string]interface{}{"some": "some"},
			schema.New(
				"a", field.String(),
				"b", field.String(),
			),
			map[string]interface{}{},
			[]string{"a", "b"},
			nil,
			false},
		{"nil object",
			map[string]interface{}{"a": "aa"},
			schema.New(
				"a", field.String(),
				"j", field.Object(
					"aa", field.String(),
					"bb", field.Number(field.NumberFormatInt),
				),
			),
			map[string]interface{}{"a": "aa"},
			[]string{"a", "j", "j.aa", "j.bb"},
			nil,
			false},
		{
			"object condition",
			map[string]interface{}{"key": "a", "object_b": map[string]interface{}{"field1": "a", "field2": "a"}},
			schema.New(
				"key", field.String(modify.Default("default")),
				"object_b", field.Object(
					"field1", field.String(),
					"field2", field.String(),
				),
				"object_a", field.Object(
					"field1", field.String(),
					"field2", field.String(),
				).SetCondition("key=='a'"),
			),
			map[string]interface{}{"key": "a", "object_b": map[string]interface{}{"field1": "a", "field2": "a"}},
			[]string{"key", "object_b", "object_a", "object_b.field1", "object_b.field2"},
			[]string{"field1", "field2"},
			false,
		},
		{
			"object condition with nil data",
			nil,
			schema.New(
				"key", field.String(modify.Default("default")),
				"object_b", field.Object(
					"field1", field.String(),
					"field2", field.String(),
				),
				"object_a", field.Object(
					"field1", field.String(),
					"field2", field.String(),
				).SetCondition("key=='a'"),
			),
			nil,
			[]string{"key", "object_b", "object_b.field1", "object_b.field2"},
			[]string{"object_a", "field1", "field2"},
			false,
		},
	}

	ctx := context.Background()
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			gotValue, gotSchema, err := tt.schema.Introspect(ctx, tt.data)
			require.NoError(t, err)

			if tt.wantErr {
				assert.Error(t, err)
			} else {
				assert.NoError(t, err)
			}

			for _, f := range tt.wantFields {
				fld := gotSchema.GetField(f)
				assert.NotNil(t, fld, fmt.Sprintf("поле '%s' должно присутствовать в схеме", f))
			}

			for _, f := range tt.dontWantFields {
				fld := gotSchema.GetField(f)
				assert.Nil(t, fld, fmt.Sprintf("поле '%s' должно отсутствовать в схеме", f))
			}

			//b, err := json.MarshalIndent(got.Schema, "", "  ")
			//require.NoError(t, err)
			//fmt.Printf("---\n%s\n---\n", b)
			assert.Equal(t, tt.want, gotValue)
		})
	}

}

func TestSchema_IntrospectObjectArray(t *testing.T) {
	tests := []struct {
		name       string
		data       map[string]interface{}
		schema     *schema.Schema
		want       map[string]interface{}
		wantParams []string
		wantErr    bool
	}{
		{
			"simple",
			map[string]interface{}{"array": []interface{}{map[string]interface{}{"field1": "a", "field2": "a"}}},
			schema.New(
				"array", field.Array(
					field.Object(
						"field1", field.String(),
						"field2", field.String(),
					),
				)),
			map[string]interface{}{"array": []interface{}{map[string]interface{}{"field1": "a", "field2": "a"}}},
			[]string{"field1", "field2"},
			false,
		},
		{
			"empty data",
			nil,
			schema.New(
				"array", field.Array(
					field.Object(
						"field1", field.String(),
						"field2", field.String(),
					),
				)),
			nil,
			[]string{"field1", "field2"},
			false,
		},
	}

	ctx := context.Background()
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			gotValue, gotSchema, err := tt.schema.Introspect(ctx, tt.data)
			require.NoError(t, err)

			if tt.wantErr {
				assert.Error(t, err)
			} else {
				assert.NoError(t, err)
			}

			for _, f := range tt.wantParams {
				fld := gotSchema.GetField("array")
				p, ok := fld.Params.(*field.ArrayParameters).Item.Params.(*field.ObjectParameters)
				assert.True(t, ok)
				assert.Contains(t, p.Fields, f, fmt.Sprintf("поле '%s' должно присутствовать в параметрах Item", f))
			}
			assert.Equal(t, tt.want, gotValue)
		})
	}

}

func TestSchema_Load(t *testing.T) {
	f1 := schema.New(
		"f", field.String(),
		"s3", field.Object("a1", field.Number(field.NumberFormatInt), "f", field.String()).WithIncludes("f2"),
	)
	f2 := schema.New("a", field.String())

	sch := schema.NewFromField(field.Object(
		"s1", field.String(),
		"s2", field.String(),
		"s3", field.Object("a1", field.String(), "a2", field.String()),
		"s4", field.Array(field.Object().WithIncludes("f2")),
	).WithIncludes("f1", "f2"),
	)

	loader := field.MultiLoader(
		field.LoaderFunc(func(ref string) (fs []*field.Field, err error) {
			if ref == "f1" {
				f := f1.Field // copy
				return []*field.Field{&f}, nil
			}
			return nil, fmt.Errorf("invalid schema reference: %s", ref)
		}),
		field.LoaderFunc(func(ref string) (fs []*field.Field, err error) {
			if ref == "f2" {
				f := f2.Field // copy
				return []*field.Field{&f}, nil
			}
			return nil, fmt.Errorf("invalid schema reference: %s", ref)
		}),
	)
	schema.SetDefaultLoader(loader)

	err := sch.Load(nil)
	require.NoError(t, err)

	//b, _ := json.MarshalIndent(sch, "", "  ")
	//fmt.Println(string(b))

	assert.NotNil(t, sch.GetField("s1"))
	assert.NotNil(t, sch.GetField("s2"))
	assert.NotNil(t, sch.GetField("f"))
	assert.NotNil(t, sch.GetField("a"))
	assert.NotNil(t, sch.GetField("s3"))
	assert.NotNil(t, sch.GetField("s3.f"))
	assert.NotNil(t, sch.GetField("s3.a"))
	{
		f := sch.GetField("s3.a1")
		require.NotNil(t, f)
		assert.Equal(t, f.GetType(), &field.StringType{})
	}
	assert.NotNil(t, sch.GetField("s4.a"))

}

func TestSchema_WithIncludesCircle(t *testing.T) {
	f1 := schema.New("f2", field.Object().WithIncludes("f2"))
	f2 := schema.New("f3", field.Object().WithIncludes("f3"))
	f3 := schema.New("f1", field.Object().WithIncludes("f1"))

	loader := field.MultiLoader(
		field.LoaderFunc(func(ref string) (fs []*field.Field, err error) {
			if ref == "f1" {
				f := f1.Field // copy
				return []*field.Field{&f}, nil
			}
			return nil, fmt.Errorf("invalid schema reference: %s", ref)
		}),
		field.LoaderFunc(func(ref string) (fs []*field.Field, err error) {
			if ref == "f2" {
				f := f2.Field // copy
				return []*field.Field{&f}, nil
			}
			return nil, fmt.Errorf("invalid schema reference: %s", ref)
		}),
		field.LoaderFunc(func(ref string) (fs []*field.Field, err error) {
			if ref == "f3" {
				f := f3.Field // copy
				return []*field.Field{&f}, nil
			}
			return nil, fmt.Errorf("invalid schema reference: %s", ref)
		}),
	)
	schema.SetDefaultLoader(loader)
	sch := schema.NewFromField(field.Object().WithIncludes("f1"))

	err := sch.Load(nil)
	require.Error(t, err)
	assert.EqualError(t, err, "limit for included fields exceeded")
}

func TestSchema_EnumUIOptions(t *testing.T) {
	schm := `{
	"type": "object",
	"params": {
		"fields": {
			"enum": {
				"options": {
					"enum": [{
							"name": "1",
							"value": "one",
							"ui" : {
							  "color": "color",
							  "icon": "icon",
							  "spin": true,
							  "blink": false		
							}
							},
							{
								"name": "2",
								"value": "two"
							}
					]
				},
				"type": "string"
			}
		}
	}
}`
	s := schema.New()
	err := json.Unmarshal([]byte(schm), s)
	require.NoError(t, err)
}
