package field

import "context"

type WalkFuncResult struct {
	Context context.Context
	Value   interface{}
	Changed bool
	Stop    bool
}

// WalkFunc - функция которая будет применена к значению каждого поля в результате обход данных функцией `Walk`
// Возвращает новое значение если таковое имеется, признак изменения данных и ошибку в случае возникновения
type WalkFunc func(ctx context.Context, fld *Field, v interface{}) (result WalkFuncResult, err error)

// FieldWalker - интерфейс позволяющий выполнить обход предоставленного поля
// Реализуется в типах
type FieldWalker interface {
	// Walk выполняет обход данных в соответствии с полем. Функция возвращает результат:
	// результат обхода, флаг что данные изменены, ошибку
	Walk(ctx context.Context, fld *Field, v interface{}, fn WalkFunc, opts *WalkOptions) (interface{}, bool, error)
}

type WalkOptions struct {
	WalkSchema bool
}

type WalkOption func(opts *WalkOptions)

// WalkSchema указывает что необходимо выполнять обход по схеме, вместо обхода по данным.
// При обходе по данным в случае отсутствия данных для поля дальнейшая обработка данного поля не выполняется.
// При обходе по схемы будут обработаны все поля присутствующие в схеме вне зависимости от наличия данных.
func WalkSchema() WalkOption {
	return func(opts *WalkOptions) {
		opts.WalkSchema = true
	}
}

func WalkOpts(o *WalkOptions) WalkOption {
	return func(opts *WalkOptions) {
		*opts = *o
	}
}

func NewWalkOptions(opt ...WalkOption) *WalkOptions {
	opts := &WalkOptions{}
	for _, o := range opt {
		o(opts)
	}
	return opts
}

// Walker - интерфейс позволяющий выполнить обход данных и содержит схему внутри
// Реализовано в Field, Schema
type Walker interface {
	Walk(ctx context.Context, v interface{}, fn WalkFunc, opt ...WalkOption) (interface{}, bool, error)
}
