package field

import (
	"context"
	"fmt"
	"time"
)

var (
	zeroTime      = time.Time{}
	timestampType = &TimestampType{}
)

type TimestampParameters struct{}

func (t TimestampParameters) Type() Type                   { return timestampType }
func (t *TimestampParameters) Clone(reset bool) Parameters { return t }

type TimestampType struct{}

func (t TimestampType) Name() string {
	return "timestamp"
}

func (TimestampType) NewParameters() Parameters {
	return &TimestampParameters{}
}

func (TimestampType) IsEmpty(v interface{}) bool {
	return v == 0 || v == nil
}

func toTimestamp(i interface{}) (interface{}, error) {
	switch v := i.(type) {
	case nil:
		return nil, nil
	case int64:
		return v, nil
	case int:
		return int64(v), nil
	case int8:
		return int64(v), nil
	case int32:
		return int64(v), nil
	case uint64:
		return int64(v), nil
	case uint:
		return int64(v), nil
	case uint8:
		return int64(v), nil
	case uint32:
		return int64(v), nil
	case float32:
		return int64(v), nil
	case float64:
		return int64(v), nil
	default:
		return nil, fmt.Errorf("unsupported value type: \"%T\"", i)
	}
}

func (TimestampType) Decode(_ context.Context, _ *Field, v interface{}) (interface{}, error) {
	switch i := v.(type) {
	case string:
		duration, err := time.ParseDuration(i)
		if err == nil {
			return duration.Nanoseconds(), nil
		}
		t, err := time.Parse(time.TimeOnly, i)

		if err == nil {
			return t.AddDate(1, 0, 0).Sub(zeroTime).Nanoseconds(), nil
		}
		return nil, err
	default:
		return toTimestamp(i)
	}
}

func (TimestampType) Encode(_ context.Context, _ *Field, v interface{}) (interface{}, error) {
	switch i := v.(type) {
	default:
		return toTimestamp(i)
	}
}

func Timestamp(o ...interface{}) *Field {
	return NewField(&TimestampParameters{}, o...)
}
