package field

import (
	"context"
	"math"

	"github.com/pkg/errors"
)

const (
	NumberFormatInt   = "int"
	NumberFormatFloat = "float"
)

var numberType = &NumberType{}

type NumberParameters struct {
	Format string `json:"format,omitempty"`
}

func (NumberParameters) Type() Type                    { return numberType }
func (p NumberParameters) Clone(reset bool) Parameters { return &p }

type NumberType struct{}

func (NumberType) Name() string {
	return "number"
}

func (NumberType) NewParameters() Parameters {
	return &NumberParameters{}
}

func (NumberType) IsEmpty(v interface{}) bool {
	return v == nil
}

func ToNumber(i interface{}) (interface{}, error) {
	switch v := i.(type) {
	case int64:
		return v, nil
	case int:
		return int64(v), nil
	case int8:
		return int64(v), nil
	case int32:
		return int64(v), nil
	case uint64:
		return v, nil
	case uint:
		return uint64(v), nil
	case uint8:
		return uint64(v), nil
	case uint32:
		return uint64(v), nil
	case float32:
		return float64(v), nil
	case float64:
		return v, nil
	}
	return 0, errors.Errorf("error convert \"%s\" to number", i)
}

func (n NumberType) Decode(ctx context.Context, field *Field, v interface{}) (interface{}, error) {
	return n.decode(ctx, field, v)
}

func (NumberType) decode(_ context.Context, field *Field, v interface{}) (interface{}, error) {
	params, ok := field.Params.(*NumberParameters)
	if !ok {
		return nil, errors.New("field parameters required")
	}

	if v == nil {
		return v, nil
	}

	n, err := ToNumber(v)
	if err != nil {
		return nil, err
	}

	switch params.Format {
	case NumberFormatInt:
		switch i := n.(type) {
		case int64:
			return i, nil
		case uint64:
			return i, nil
		case float64:
			return int64(math.Round(i)), nil
		}
	case NumberFormatFloat:
		switch i := n.(type) {
		case float64:
			return i, nil
		case int64:
			return float64(i), nil
		case uint64:
			return float64(i), nil
		}
	}
	return n, nil
}

func (n NumberType) Encode(ctx context.Context, field *Field, v interface{}) (interface{}, error) {
	return n.decode(ctx, field, v)
}

func Number(format string, o ...interface{}) *Field {
	return NewField(&NumberParameters{Format: format}, o...)
}
