package middleware

import (
	"context"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	service "git.perx.ru/perxis/perxis-go/pkg/roles"
)

func makeKey(ss ...string) string {
	return strings.Join(ss, "-")
}

func CachingMiddleware(cache *cache.Cache) Middleware {
	return func(next service.Roles) service.Roles {
		return &cachingMiddleware{
			cache: cache,
			next:  next,
		}
	}
}

type cachingMiddleware struct {
	cache *cache.Cache
	next  service.Roles
}

func (m cachingMiddleware) Create(ctx context.Context, role *service.Role) (rl *service.Role, err error) {
	rl, err = m.next.Create(ctx, role)
	if err == nil {
		m.cache.Remove(rl.SpaceID)
	}
	return rl, err
}

func (m cachingMiddleware) Get(ctx context.Context, spaceId string, roleId string) (rl *service.Role, err error) {
	key := makeKey(spaceId, roleId)
	value, e := m.cache.Get(key)
	if e == nil {
		return value.(*service.Role), err
	}
	rl, err = m.next.Get(ctx, spaceId, roleId)
	if err == nil {
		m.cache.Set(key, rl)
	}
	return rl, err
}

func (m cachingMiddleware) List(ctx context.Context, spaceId string) (roles []*service.Role, err error) {
	value, e := m.cache.Get(spaceId)
	if e == nil {
		return value.([]*service.Role), err
	}
	roles, err = m.next.List(ctx, spaceId)
	if err == nil {
		m.cache.Set(spaceId, roles)
	}
	return roles, err
}

func (m cachingMiddleware) Update(ctx context.Context, role *service.Role) (err error) {
	err = m.next.Update(ctx, role)
	if err == nil {
		key := makeKey(role.SpaceID, role.ID)
		m.cache.Remove(key)
		m.cache.Remove(role.SpaceID)
	}
	return err
}

func (m cachingMiddleware) Delete(ctx context.Context, spaceId string, roleId string) (err error) {
	err = m.next.Delete(ctx, spaceId, roleId)
	if err == nil {
		key := makeKey(spaceId, roleId)
		m.cache.Remove(key)
		m.cache.Remove(spaceId)
	}
	return err
}
