package references

import (
	"context"
	"fmt"
	"reflect"

	"git.perx.ru/perxis/perxis-go/pkg/data"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/items"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
)

const ReferenceTypeName = "reference"

type ReferenceParameters struct {
	AllowedCollections []string `json:"allowedCollections"`
}

func (p ReferenceParameters) Type() field.Type { return &ReferenceType{} }

func (p ReferenceParameters) Clone(reset bool) field.Parameters {
	if p.AllowedCollections != nil {
		cols := make([]string, 0, len(p.AllowedCollections))
		for _, c := range p.AllowedCollections {
			cols = append(cols, c)
		}
		p.AllowedCollections = cols
	}
	return &p
}

type ReferenceType struct{}

func NewReferenceType() *ReferenceType {
	return &ReferenceType{}
}

func (t ReferenceType) Name() string { return ReferenceTypeName }

func (t *ReferenceType) NewParameters() field.Parameters {
	return &ReferenceParameters{}
}

func (t ReferenceType) Decode(_ context.Context, fld *field.Field, v interface{}) (interface{}, error) {
	if v == nil {
		return nil, nil
	}

	r, ok := v.(map[string]interface{})
	if !ok {
		return nil, errors.Errorf("ReferenceField decode error: incorrect type: \"%s\", expected \"map[string]interface{}\"", reflect.ValueOf(v).Kind())
	}
	id, ok1 := r["id"].(string)
	if !ok1 || id == "" {
		return nil, errors.Errorf("ReferenceField decode error: field \"id\" required")
	}
	collID, ok2 := r["collection_id"].(string)
	if !ok2 || collID == "" {
		return nil, errors.Errorf("ReferenceField decode error: field \"collection_id\" required")
	}
	disabled, _ := r["disabled"].(bool)

	return &Reference{ID: id, CollectionID: collID, Disabled: disabled}, nil
}

func (t ReferenceType) Encode(_ context.Context, fld *field.Field, v interface{}) (interface{}, error) {
	if v == nil {
		return nil, nil
	}

	val, ok := v.(*Reference)

	if !ok {
		return nil, errors.Errorf("ReferenceField encode error: incorrect type: \"%s\", expected \"*Reference\"", reflect.ValueOf(v).Kind())
	}
	if val == nil {
		return nil, nil
	}
	ref := map[string]interface{}{
		"id":            val.ID,
		"collection_id": val.CollectionID,
		"disabled":      val.Disabled,
	}
	return ref, nil
}

func (t *ReferenceType) PreSave(ctx context.Context, f *field.Field, v interface{}, itemCtx *items.Context) (interface{}, bool, error) {
	params, ok := f.Params.(*ReferenceParameters)
	if !ok {
		return nil, false, errors.New("field parameters required")
	}

	if v == nil {
		return nil, false, nil
	}
	ref, ok := v.(*Reference)
	if !ok {
		return nil, false, fmt.Errorf("ReferenceField presave error: incorrect type: \"%s\", expected \"*Reference\"", reflect.ValueOf(v).Kind())
	}

	// Проверка на наличие ссылок заданных типов
	if len(params.AllowedCollections) > 0 {
		ok := false
		for _, allowed := range params.AllowedCollections {

			if data.GlobMatch(ref.CollectionID, allowed) {
				ok = true
				break
			}
		}
		if !ok {
			return nil, false, errors.Errorf("usage of collection \"%s\" not allowed", ref.CollectionID)
		}
	}

	return ref, false, nil
}

// Field - создает новое поле Field типа ReferenceType
// ReferenceType должен быть предварительно создан через `NewReferenceType` и зарегистрирован `field.Register`
func Field(allowedColls []string, o ...interface{}) *field.Field {
	_, ok := field.GetType(ReferenceTypeName)
	if !ok {
		panic("field reference type not registered")
	}

	return field.NewField(&ReferenceParameters{AllowedCollections: allowedColls}, o...)
}

func (t *ReferenceType) IsEmpty(v interface{}) bool {
	if v == nil {
		return true
	}

	ref, ok := v.(*Reference)

	return !ok || ref.ID == "" && ref.CollectionID == ""
}

func init() {
	field.Register(NewReferenceType())
}
