package queue

import (
	"runtime"
	"sync"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/id"
)

const (
	defaultSize            = 100
	defaultStoreResultsTTL = 1 * time.Hour
)

type Waiter interface {
	Wait(string) (*JobResult, error)
}

type Job func() error

type JobResult struct {
	Err error
}

type JobGroup struct {
	wg sync.WaitGroup
}

func (g *JobGroup) Add(j Job) Job {
	g.wg.Add(1)
	return func() error {
		defer g.wg.Done()
		return j()
	}
}

func (g *JobGroup) Wait() {
	g.wg.Wait()
}

// Queue предназначена для постановки в очередь и выполнения задач с соблюдением
// максимально возможного числа одновременно выполняемых задач (по умолчанию число
// логических CPU). Каждой задаче присваивается идентификатор, по которому можно
// получить результат выполнения задачи.
type Queue struct {
	jobsCh          chan Job
	results         sync.Map
	StoreResultsTTL time.Duration
	serveWG         sync.WaitGroup
	done            chan struct{}
	NumWorkers      int
	Size            int
}

// AddJob - добавить задачу в очередь на обработку. В случае, если в очереди скопилось задач
// больше, чем defaultSize, вернется ошибка и задача не будет добавлена. Каждой задаче
// присваивается идентификатор, по которому можно получить результат выполнения задачи. Все результаты
// доступны в течение часа после завершения выполнения задачи.
func (j *Queue) AddJob(job Job) (jobID string, err error) {
	if j == nil {
		return
	}

	jobID = id.GenerateNewID()
	resCh := make(chan *JobResult, 1)

	trackedJob := func() error {
		err := job()
		resCh <- &JobResult{Err: err}
		close(resCh)

		go func() {
			select {
			case <-j.done:
				return
			case <-time.After(j.StoreResultsTTL):
				j.results.Delete(jobID)
			}
		}()

		return err
	}

	// нужно добавить до того, как задача попадет в очередь
	j.results.Store(jobID, resCh)

	select {
	case j.jobsCh <- trackedJob:
	default:
		j.results.Delete(jobID)
		return "", errors.New("queue size exceeded")
	}

	return jobID, nil
}

func (j *Queue) Wait(jobID string) (*JobResult, error) {
	resCh, ok := j.results.Load(jobID)
	if !ok {
		return nil, errors.Errorf("job '%s' not found", jobID)
	}
	res := <-resCh.(chan *JobResult)
	return res, nil
}

func (j *Queue) WaitCh(jobID string) (<-chan *JobResult, error) {
	resCh, ok := j.results.Load(jobID)
	if !ok {
		return nil, errors.Errorf("job '%s' not found", jobID)
	}
	return resCh.(chan *JobResult), nil
}

func (j *Queue) IsStarted() bool {
	return j != nil && j.jobsCh != nil && j.done != nil
}

func (j *Queue) Start() {
	if j == nil {
		panic("job runner not created")
	}

	if j.jobsCh != nil || j.done != nil {
		return
	}

	if j.Size == 0 {
		j.Size = defaultSize
	}

	j.jobsCh = make(chan Job, j.Size)
	j.done = make(chan struct{})
	j.serveWG = sync.WaitGroup{}

	if j.StoreResultsTTL == 0 {
		j.StoreResultsTTL = defaultStoreResultsTTL
	}

	if j.NumWorkers == 0 {
		j.NumWorkers = runtime.NumCPU()
	}

	j.serveWG.Add(j.NumWorkers)
	for i := 0; i < j.NumWorkers; i++ {
		go j.worker()
	}
}

func (j *Queue) Stop() {
	if j.done == nil && j.jobsCh == nil {
		return
	}

	close(j.done)
	j.serveWG.Wait()
	close(j.jobsCh)
	j.done = nil
	j.jobsCh = nil
}

func (j *Queue) worker() {
	defer j.serveWG.Done()

	for {
		select {
		case job, ok := <-j.jobsCh:
			if !ok {
				return // channel closed
			}
			_ = job()
		case <-j.done:
			return
		}
	}
}
