package options

import "time"

// SortOptions настройки сортировки результатов
type SortOptions struct {
	Sort []string
}

// PaginationOptions настройки возвращаемых страниц результатов
type PaginationOptions struct {
	PageNum  int
	PageSize int
}

// FieldOptions настройки включения/исключения полей из результатов запроса
type FieldOptions struct {
	// Fields - Наименования полей для включения/исключения из результатов запроса (только указанные поля)
	// Если `ExcludeFields` не установлен, то результат содержит только указанные поля
	// Если `ExcludeFields` установлен, то результат содержит все поля кроме указанных
	Fields []string

	// ExcludeFields- Если флаг установлен, то перечисленные поля `Fields` следует исключить из результатов
	ExcludeFields bool
}

// FindOptions настройки возвращаемых результатов поиска
type FindOptions struct {
	SortOptions
	PaginationOptions
	FieldOptions
}

// NewFindOptions создает новые результаты поиска
func NewFindOptions(pageNum, pageSize int, sort ...string) *FindOptions {
	return &FindOptions{
		PaginationOptions: PaginationOptions{
			PageNum:  pageNum,
			PageSize: pageSize,
		},
		SortOptions: SortOptions{
			Sort: sort,
		},
	}
}

// MergeFindOptions объединяет в FindOptions различные варианты настроек
func MergeFindOptions(opts ...interface{}) *FindOptions {
	fo := &FindOptions{}
	for _, opt := range opts {
		if opt == nil {
			continue
		}

		switch o := opt.(type) {
		case FindOptions:
			fo.SortOptions = MergeSortOptions(fo.SortOptions, o.SortOptions)
			fo.PaginationOptions = MergePaginationOptions(fo.PaginationOptions, o.PaginationOptions)
			fo.FieldOptions = MergeFieldOptions(fo.FieldOptions, o.FieldOptions)
		case *FindOptions:
			fo.SortOptions = MergeSortOptions(fo.SortOptions, o.SortOptions)
			fo.PaginationOptions = MergePaginationOptions(fo.PaginationOptions, o.PaginationOptions)
			fo.FieldOptions = MergeFieldOptions(fo.FieldOptions, o.FieldOptions)
		case SortOptions:
			fo.SortOptions = MergeSortOptions(fo.SortOptions, o)
		case *SortOptions:
			fo.SortOptions = MergeSortOptions(fo.SortOptions, *o)
		case PaginationOptions:
			fo.PaginationOptions = MergePaginationOptions(fo.PaginationOptions, o)
		case *PaginationOptions:
			fo.PaginationOptions = MergePaginationOptions(fo.PaginationOptions, *o)
		case FieldOptions:
			fo.FieldOptions = o
		case *FieldOptions:
			fo.FieldOptions = *o
		}
	}
	return fo
}

type TimeFilter struct {
	Before, After time.Time
}

// MergeSortOptions объединяет настройки сортировки
func MergeSortOptions(options ...SortOptions) SortOptions {
	fo := SortOptions{}
	for _, opt := range options {
		if len(opt.Sort) == 0 {
			continue
		}
		fo.Sort = append(fo.Sort, opt.Sort...)
	}
	return fo
}

// MergePaginationOptions объединяет настройки страниц
func MergePaginationOptions(options ...PaginationOptions) PaginationOptions {
	fo := PaginationOptions{}
	for _, opt := range options {
		if opt.PageSize == 0 && opt.PageNum == 0 {
			continue
		}
		fo.PageNum = opt.PageNum
		fo.PageSize = opt.PageSize
	}
	return fo
}

// MergeFieldOptions выполняет слияние опций для возвращаемых полей.
// Выбирается не пустое значение.
func MergeFieldOptions(options ...FieldOptions) FieldOptions {
	fo := FieldOptions{}
	for _, opt := range options {
		if len(opt.Fields) > 0 {
			fo.Fields = opt.Fields
			fo.ExcludeFields = opt.ExcludeFields
			return fo
		}
	}
	return fo
}
