package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/members"
	mocksmembers "git.perx.ru/perxis/perxis-go/pkg/members/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestMembersCache(t *testing.T) {

	const (
		orgId  = "orgId"
		userId = "userId"
		size   = 5
		ttl    = 20 * time.Millisecond
	)

	ctx := context.Background()

	t.Run("Get from cache", func(t *testing.T) {
		mbrs := &mocksmembers.Members{}
		svc := CachingMiddleware(cache.NewCache(size, ttl))(mbrs)

		mbrs.On("Get", mock.Anything, orgId, userId).Return(members.RoleOwner, nil).Once()

		v1, err := svc.Get(ctx, orgId, userId)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, orgId, userId)
		require.NoError(t, err)
		assert.Equal(t, v1, v2, "Ожидается получение объекта из кэша, после повторного запроса get.")

		mbrs.AssertExpectations(t)
	})

	t.Run("Invalidate cache", func(t *testing.T) {
		t.Run("After Set", func(t *testing.T) {
			mbrs := &mocksmembers.Members{}
			svc := CachingMiddleware(cache.NewCache(size, ttl))(mbrs)

			mbrs.On("Get", mock.Anything, orgId, userId).Return(members.RoleOwner, nil).Once()

			v1, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)
			assert.Equal(t, v1, v2, "Ожидается получение объекта из кэша.")

			mbrs.On("Set", mock.Anything, orgId, userId, members.RoleMember).Return(nil).Once()

			err = svc.Set(ctx, orgId, userId, members.RoleMember)
			require.NoError(t, err)

			mbrs.On("Get", mock.Anything, orgId, userId).Return(members.RoleMember, nil).Once()

			v3, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)
			assert.NotEqual(t, v2, v3, "Ожидается удаление объекта из кэша и получение заново из сервиса.")
			mbrs.AssertExpectations(t)
		})

		t.Run("After Remove", func(t *testing.T) {
			mbrs := &mocksmembers.Members{}
			svc := CachingMiddleware(cache.NewCache(size, ttl))(mbrs)

			mbrs.On("Get", mock.Anything, orgId, userId).Return(members.RoleOwner, nil).Once()

			v1, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)
			assert.Equal(t, v1, v2, "Ожидается получение объекта из кэша.")

			mbrs.On("Remove", mock.Anything, orgId, userId).Return(nil).Once()

			err = svc.Remove(ctx, orgId, userId)
			require.NoError(t, err)

			mbrs.On("Get", mock.Anything, orgId, userId).Return(members.NotMember, nil).Once()

			v3, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)
			assert.NotEqual(t, v2, v3, "Ожидается удаление объекта из кэша после удаления из хранилища и получение заново из сервиса.")

			mbrs.AssertExpectations(t)
		})

		t.Run("After RemoveAll", func(t *testing.T) {
			mbrs := &mocksmembers.Members{}
			svc := CachingMiddleware(cache.NewCache(size, ttl))(mbrs)

			mbrs.On("Get", mock.Anything, orgId, userId).Return(members.RoleOwner, nil).Once()
			mbrs.On("ListMembers", mock.Anything, orgId).Return([]*members.Member{{OrgId: orgId, UserId: userId, Role: members.RoleOwner}}, nil)

			v1, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)
			assert.Equal(t, v1, v2, "Ожидается получение объекта из кэша.")

			mbrs.On("RemoveAll", mock.Anything, orgId).Return(nil).Once()

			err = svc.RemoveAll(ctx, orgId)
			require.NoError(t, err)

			mbrs.On("Get", mock.Anything, orgId, userId).Return(members.NotMember, nil).Once()

			v3, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)
			assert.NotEqual(t, v2, v3, "Ожидается удаление объекта из кэша после удаления из хранилища и получение заново из сервиса.")

			mbrs.AssertExpectations(t)
		})

		t.Run("After TTL expired", func(t *testing.T) {
			mbrs := &mocksmembers.Members{}
			svc := CachingMiddleware(cache.NewCache(size, ttl))(mbrs)

			mbrs.On("Get", mock.Anything, orgId, userId).Return(members.RoleOwner, nil).Once()

			v1, err := svc.Get(ctx, orgId, userId)

			v2, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)
			assert.Equal(t, v1, v2, "Ожидается получение объекта из кэша.")

			time.Sleep(2 * ttl)

			mbrs.On("Get", mock.Anything, orgId, userId).Return(members.RoleMember, nil).Once()

			v3, err := svc.Get(ctx, orgId, userId)
			require.NoError(t, err)
			assert.NotEqual(t, v2, v3, "Ожидается удаление объекта из кэша после истечения ttl и получение заново из сервиса.")
			mbrs.AssertExpectations(t)
		})
	})
}
