package middleware

import (
	"context"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	service "git.perx.ru/perxis/perxis-go/pkg/members"
)

func makeKey(ss ...string) string {
	return strings.Join(ss, "-")
}

func CachingMiddleware(cache *cache.Cache) Middleware {
	return func(next service.Members) service.Members {
		return &cachingMiddleware{
			cache: cache,
			next:  next,
		}
	}
}

type cachingMiddleware struct {
	cache *cache.Cache
	next  service.Members
}

func (m cachingMiddleware) Set(ctx context.Context, orgId string, userId string, role service.Role) (err error) {

	err = m.next.Set(ctx, orgId, userId, role)
	if err == nil {
		m.cache.Remove(makeKey(orgId, userId))
		m.cache.Remove(makeKey(orgId))
		m.cache.Remove(makeKey(userId))
	}
	return err
}

func (m cachingMiddleware) Get(ctx context.Context, orgId string, userId string) (role service.Role, err error) {

	key := makeKey(orgId, userId)
	value, e := m.cache.Get(key)
	if e == nil {
		return value.(service.Role), err
	}
	role, err = m.next.Get(ctx, orgId, userId)
	if err == nil {
		m.cache.Set(key, role)
	}
	return role, err
}

func (m cachingMiddleware) Remove(ctx context.Context, orgId string, userId string) (err error) {

	err = m.next.Remove(ctx, orgId, userId)
	if err == nil {
		m.cache.Remove(makeKey(orgId, userId))
		m.cache.Remove(makeKey(orgId))
		m.cache.Remove(makeKey(userId))
	}
	return err
}

func (m cachingMiddleware) RemoveAll(ctx context.Context, orgId string) (err error) {

	err = m.next.RemoveAll(ctx, orgId)
	if err == nil {
		members, _ := m.ListMembers(ctx, orgId)
		for _, member := range members {
			m.cache.Remove(member.UserId)
			m.cache.Remove(makeKey(orgId, member.UserId))
		}
	}
	return err
}

func (m cachingMiddleware) ListMembers(ctx context.Context, orgId string) (members []*service.Member, err error) {

	value, e := m.cache.Get(makeKey(orgId))
	if e == nil {
		return value.([]*service.Member), err
	}
	members, err = m.next.ListMembers(ctx, orgId)
	if err == nil {
		m.cache.Set(makeKey(orgId), members)
	}
	return members, err
}

func (m cachingMiddleware) ListOrganizations(ctx context.Context, userId string) (members []*service.Member, err error) {

	value, e := m.cache.Get(makeKey(userId))
	if e == nil {
		return value.([]*service.Member), err
	}
	members, err = m.next.ListOrganizations(ctx, userId)
	if err == nil {
		m.cache.Set(makeKey(userId), members)
	}
	return members, err
}
