package items

import (
	"context"

	"git.perx.ru/perxis/perxis-go/pkg/collections"
)

type Storage interface {
	// Items
	// ---------------------------------------------------------------------------------------------
	// Все пользовательские коллекции Perxis хранятся вместе и данные могут быть выбраны одновременно
	// из разных коллекций.
	//
	// Для работы с Items используются 3 коллекции MongoDB между которыми записи переносятся в зависимости
	// от их статуса.
	// - 'items' - рабочие записи с которыми работает API управления контентом. В коллекции находятся
	//		только текущие ревизии записи
	// - 'revisions' - все ревизии записи за исключением текущей. Ревизии записей переносятся в данную
	//		коллекцию при создании новой ревизии записи. Поиск данных по коллекции не производится
	// - 'published' - опубликованные записи, в данную коллекцию копируются записи из `items` при
	//		публикации. В коллекции находится только одна опубликованная ревизия. Коллекция обеспечивает
	//		быстрый поиск, поэтому для данных строятся Spare-индексы с использованием Схем
	// - 'archived' - архивные записи, в данную коллекцию записи переносятся при архивировании из `items`.
	//		Копии записей из `published` при этом удаляются.
	// ---------------------------------------------------------------------------------------------

	// Create - создать новую запись
	Create(ctx context.Context, coll *collections.Collection, item *Item, options ...*CreateOptions) (*Item, error)

	// CreateRevision - перенести ревизию в коллекцию Revisions
	CreateRevision(ctx context.Context, spaceId, envId, collectionId, itemId string) error

	// Update - Обновление текущей ревизии или создание новой, если в опциях передан флаг `ReplacePublishedRevision`
	Update(ctx context.Context, coll *collections.Collection, item *Item, options ...*UpdateOptions) error

	// Find - поиск записей по рабочим записям, коллекция 'items'
	Find(ctx context.Context, coll *collections.Collection, filter *Filter, opts ...*FindOptions) ([]*Item, int, error)

	// GetRevision - поиск одной ревизии одной записи
	GetRevision(ctx context.Context, coll *collections.Collection, itemId, revisionId string, options ...*GetRevisionOptions) (*Item, error)

	// ListRevisions - поиск всех ревизий одной записи
	ListRevisions(ctx context.Context, coll *collections.Collection, itemId string, options ...*ListRevisionsOptions) ([]*Item, error)

	// ChangeRevisionsItemID - заменить ID элемента у его ревизий
	ChangeRevisionsItemID(ctx context.Context, spaceId, envId, collectionId, itemId, newItemId string) error

	// Publish - опубликовать запись
	Publish(ctx context.Context, published *Item, options ...*PublishOptions) error

	// Unpublish - отменить публикацию записи
	Unpublish(ctx context.Context, unpublished *Item, options ...*UnpublishOptions) error

	// FindPublished - поиск по опубликованным записям, коллекция 'items_published'
	FindPublished(ctx context.Context, coll *collections.Collection, filter *Filter, opts ...*FindPublishedOptions) ([]*Item, int, error)

	// Archive - архивация записи
	Archive(ctx context.Context, archived *Item, options ...*ArchiveOptions) error

	// Unarchive - разархивация записи
	Unarchive(ctx context.Context, unarchived *Item, options ...*UnarchiveOptions) error

	// FindArchived - поиск по архивированным записям, коллекция 'items_archived'
	FindArchived(ctx context.Context, coll *collections.Collection, filter *Filter, opts ...*FindArchivedOptions) ([]*Item, int, error)

	// RemoveItems - удаление записи из коллекций Items
	RemoveItems(ctx context.Context, spaceId, envId, collectionId, itemId string, options ...*DeleteOptions) error

	// RemovePublished - удаление записи из коллекций Published
	RemovePublished(ctx context.Context, spaceId, envId, collectionId, itemId string, options ...*DeleteOptions) error

	// RemoveRevisions - удаление записи из всех ревизий элемента
	RemoveRevisions(ctx context.Context, spaceId, envId, collectionId, itemId string, options ...*DeleteOptions) error

	// RemoveRevision - удаление конкретной ревизии из Revisions
	RemoveRevision(ctx context.Context, spaceId, envId, collectionId, revision string, options ...*DeleteOptions) error

	// RemoveArchived - удаление записи из архива
	RemoveArchived(ctx context.Context, spaceId, envId, collectionId, itemId string, options ...*DeleteOptions) error

	// Copy - копирование записей
	Copy(ctx context.Context, src, dst *collections.Collection, itemSets ...string) error

	Reset(ctx context.Context, spaceId, envId, collectionId string) error
	Init(ctx context.Context, collection *collections.Collection) error

	// Aggregate выполняет агрегацию данных
	Aggregate(ctx context.Context, coll *collections.Collection, filter *Filter, options ...*AggregateOptions) (result map[string]interface{}, err error)
	// AggregatePublished выполняет агрегацию опубликованных данных
	AggregatePublished(ctx context.Context, coll *collections.Collection, filter *Filter, options ...*AggregatePublishedOptions) (result map[string]interface{}, err error)
}
