package middleware

import (
	"context"

	"git.perx.ru/perxis/perxis-go/pkg/collections"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/items"
	"git.perx.ru/perxis/perxis-go/pkg/schema"
)

// ClientEncodeMiddleware выполняет операции encode/decode для передаваемых данных
func ClientEncodeMiddleware(colls collections.Collections) Middleware {
	return func(items items.Items) items.Items {
		return &encodeDecodeMiddleware{
			next:  items,
			colls: colls,
		}

	}
}

type encodeDecodeMiddleware struct {
	next  items.Items
	colls collections.Collections
}

func (m *encodeDecodeMiddleware) Introspect(ctx context.Context, item *items.Item, opts ...*items.IntrospectOptions) (itm *items.Item, sch *schema.Schema, err error) {
	coll, err := m.colls.Get(ctx, item.SpaceID, item.EnvID, item.CollectionID)
	if err != nil {
		return nil, nil, err
	}

	if item, err = item.Encode(ctx, coll.Schema); err != nil {
		return
	}

	itm, sch, err = m.next.Introspect(ctx, item, opts...)
	if itm != nil && sch != nil {
		var err error
		if itm, err = itm.Decode(ctx, sch); err != nil {
			return nil, nil, err
		}
	}
	return itm, sch, err

}

func (m *encodeDecodeMiddleware) Create(ctx context.Context, item *items.Item, opts ...*items.CreateOptions) (created *items.Item, err error) {

	var col *collections.Collection

	if item != nil && (item.Data != nil || item.Translations != nil) {

		col, err = m.colls.Get(ctx, item.SpaceID, item.EnvID, item.CollectionID)
		if err != nil {
			return nil, err
		}

		if item, err = item.Encode(ctx, col.Schema); err != nil {
			return nil, err
		}
	}

	res, err := m.next.Create(ctx, item, opts...)
	if err == nil && (res.Data != nil || res.Translations != nil) {

		if col == nil {
			col, err = m.colls.Get(ctx, item.SpaceID, item.EnvID, item.CollectionID)
			if err != nil {
				return nil, err
			}
		}

		res, err = res.Decode(ctx, col.Schema)
	}

	return res, err
}

func (m *encodeDecodeMiddleware) Update(ctx context.Context, upd *items.Item, options ...*items.UpdateOptions) (err error) {
	var col *collections.Collection
	if upd != nil && (upd.Data != nil || upd.Translations != nil) {
		col, err = m.colls.Get(ctx, upd.SpaceID, upd.EnvID, upd.CollectionID)
		if err != nil {
			return err
		}
		if upd, err = upd.Encode(ctx, col.Schema); err != nil {
			return err
		}
	}
	return m.next.Update(ctx, upd, options...)
}

func (m *encodeDecodeMiddleware) Find(ctx context.Context, spaceId, envId, collectionId string, filter *items.Filter, options ...*items.FindOptions) (items []*items.Item, total int, err error) {
	items, total, err = m.next.Find(ctx, spaceId, envId, collectionId, filter, options...)
	if err == nil && total > 0 {
		col, err := m.colls.Get(ctx, spaceId, envId, collectionId)
		if err != nil {
			return nil, 0, err
		}
		for i, itm := range items {
			itm, err = itm.Decode(ctx, col.Schema)
			if err != nil {
				return nil, 0, err
			}

			items[i] = itm
		}
	}
	return
}

func (m *encodeDecodeMiddleware) Get(ctx context.Context, spaceId, envId, collectionId, itemId string, options ...*items.GetOptions) (item *items.Item, err error) {
	item, err = m.next.Get(ctx, spaceId, envId, collectionId, itemId, options...)
	if err == nil && item != nil {
		col, err := m.colls.Get(ctx, spaceId, envId, collectionId)
		if err != nil {
			return nil, err
		}
		item, err = item.Decode(ctx, col.Schema)
		if err != nil {
			return nil, err

		}
	}
	return
}

func (m *encodeDecodeMiddleware) Publish(ctx context.Context, item *items.Item, opts ...*items.PublishOptions) (err error) {
	if item != nil && (item.Data != nil || item.Translations != nil) {
		col, err := m.colls.Get(ctx, item.SpaceID, item.EnvID, item.CollectionID)
		if err != nil {
			return err
		}

		if item, err = item.Encode(ctx, col.Schema); err != nil {
			return err
		}
	}

	return m.next.Publish(ctx, item, opts...)
}

func (m *encodeDecodeMiddleware) Unpublish(ctx context.Context, item *items.Item, opts ...*items.UnpublishOptions) (err error) {
	if item != nil && (item.Data != nil || item.Translations != nil) {
		col, err := m.colls.Get(ctx, item.SpaceID, item.EnvID, item.CollectionID)
		if err != nil {
			return err
		}

		if item, err = item.Encode(ctx, col.Schema); err != nil {
			return err
		}
	}

	return m.next.Unpublish(ctx, item, opts...)
}

func (m *encodeDecodeMiddleware) GetPublished(ctx context.Context, spaceId, envId, collectionId, itemId string, options ...*items.GetPublishedOptions) (item *items.Item, err error) {
	item, err = m.next.GetPublished(ctx, spaceId, envId, collectionId, itemId, options...)
	if err == nil && item != nil {
		col, err := m.colls.Get(ctx, spaceId, envId, collectionId)
		if err != nil {
			return nil, err
		}
		item, err = item.Decode(ctx, col.Schema)
		if err != nil {
			return nil, err

		}
	}
	return
}

func (m *encodeDecodeMiddleware) FindPublished(ctx context.Context, spaceId, envId, collectionId string, filter *items.Filter, options ...*items.FindPublishedOptions) (items []*items.Item, total int, err error) {
	items, total, err = m.next.FindPublished(ctx, spaceId, envId, collectionId, filter, options...)
	if err == nil && total > 0 {
		col, err := m.colls.Get(ctx, spaceId, envId, collectionId)
		if err != nil {
			return nil, 0, err
		}
		for i, itm := range items {
			itm, err = itm.Decode(ctx, col.Schema)
			if err != nil {
				return nil, 0, err
			}

			items[i] = itm
		}
	}
	return
}

func (m *encodeDecodeMiddleware) GetRevision(ctx context.Context, spaceId, envId, collectionId, itemId, revisionId string, options ...*items.GetRevisionOptions) (item *items.Item, err error) {
	item, err = m.next.GetRevision(ctx, spaceId, envId, collectionId, itemId, revisionId, options...)
	if err == nil && item != nil {
		col, err := m.colls.Get(ctx, spaceId, envId, collectionId)
		if err != nil {
			return nil, err
		}
		item, err = item.Decode(ctx, col.Schema)
		if err != nil {
			return nil, err

		}
	}
	return
}

func (m *encodeDecodeMiddleware) ListRevisions(ctx context.Context, spaceId, envId, collectionId, itemId string, options ...*items.ListRevisionsOptions) (items []*items.Item, err error) {
	items, err = m.next.ListRevisions(ctx, spaceId, envId, collectionId, itemId, options...)
	if err == nil && len(items) > 0 {
		col, err := m.colls.Get(ctx, spaceId, envId, collectionId)
		if err != nil {
			return nil, err
		}
		for i, itm := range items {
			itm, err = itm.Decode(ctx, col.Schema)
			if err != nil {
				return nil, err
			}

			items[i] = itm
		}
	}
	return
}

func (m *encodeDecodeMiddleware) FindArchived(ctx context.Context, spaceId, envId, collectionId string, filter *items.Filter, options ...*items.FindArchivedOptions) (items []*items.Item, total int, err error) {
	items, total, err = m.next.FindArchived(ctx, spaceId, envId, collectionId, filter, options...)
	if err == nil && total > 0 {
		col, err := m.colls.Get(ctx, spaceId, envId, collectionId)
		if err != nil {
			return nil, 0, err
		}
		for i, itm := range items {
			itm, err = itm.Decode(ctx, col.Schema)
			if err != nil {
				return nil, 0, err
			}

			items[i] = itm
		}
	}
	return
}

func (m *encodeDecodeMiddleware) Archive(ctx context.Context, item *items.Item, opts ...*items.ArchiveOptions) (err error) {
	if item != nil && (item.Data != nil || item.Translations != nil) {
		col, err := m.colls.Get(ctx, item.SpaceID, item.EnvID, item.CollectionID)
		if err != nil {
			return err
		}

		if item, err = item.Encode(ctx, col.Schema); err != nil {
			return err
		}
	}

	return m.next.Archive(ctx, item, opts...)
}

func (m *encodeDecodeMiddleware) Unarchive(ctx context.Context, item *items.Item, opts ...*items.UnarchiveOptions) (err error) {
	if item != nil && (item.Data != nil || item.Translations != nil) {
		col, err := m.colls.Get(ctx, item.SpaceID, item.EnvID, item.CollectionID)
		if err != nil {
			return err
		}

		if item, err = item.Encode(ctx, col.Schema); err != nil {
			return err
		}
	}

	return m.next.Unarchive(ctx, item, opts...)
}

func (m *encodeDecodeMiddleware) Delete(ctx context.Context, item *items.Item, options ...*items.DeleteOptions) (err error) {
	return m.next.Delete(ctx, item, options...)
}

func (m *encodeDecodeMiddleware) Undelete(ctx context.Context, item *items.Item, options ...*items.UndeleteOptions) (err error) {
	return m.next.Undelete(ctx, item, options...)
}

func (m *encodeDecodeMiddleware) Aggregate(ctx context.Context, spaceId, envId, collectionId string, filter *items.Filter, options ...*items.AggregateOptions) (result map[string]interface{}, err error) {
	res, err := m.next.Aggregate(ctx, spaceId, envId, collectionId, filter, options...)
	if len(res) > 0 && len(options) > 0 {
		col, err := m.colls.Get(ctx, spaceId, envId, collectionId)
		if err != nil {
			return nil, errors.Wrap(err, "encode aggregate result")
		}
		o := items.MergeAggregateOptions(options...)
		res, err = items.DecodeAggregateResult(ctx, o.Fields, res, col.Schema)
		if err != nil {
			return nil, errors.Wrap(err, "encode aggregate result")
		}
	}
	return res, err
}

func (m *encodeDecodeMiddleware) AggregatePublished(ctx context.Context, spaceId, envId, collectionId string, filter *items.Filter, options ...*items.AggregatePublishedOptions) (result map[string]interface{}, err error) {
	res, err := m.next.AggregatePublished(ctx, spaceId, envId, collectionId, filter, options...)
	if len(res) > 0 && len(options) > 0 {
		col, err := m.colls.Get(ctx, spaceId, envId, collectionId)
		if err != nil {
			return nil, errors.Wrap(err, "get collection")
		}
		o := items.MergeAggregatePublishedOptions(options...)
		res, err = items.DecodeAggregateResult(ctx, o.Fields, res, col.Schema)
		if err != nil {
			return nil, err
		}
	}
	return res, err
}
