package items

import (
	"fmt"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"github.com/stretchr/testify/assert"
)

func TestItem_Set(t *testing.T) {
	item := &Item{}

	item.Set("id", "id")
	assert.Equal(t, "id", item.ID)
	now := time.Now()

	item.Set("created_at", now)
	assert.Equal(t, now, item.CreatedAt)

	item.Set("a.b.c", 101)
	assert.Equal(t, map[string]any{"a": map[string]any{"b": map[string]any{"c": 101}}}, item.Data)

}

func TestGetField(t *testing.T) {
	sch := schema.New(
		"a", field.String(),
		"obj", field.Object(
			"a", field.Number(field.NumberFormatFloat),
			"b", field.String(),
		),
		"arr", field.Array(field.Object("a", field.Time())),
	)

	tests := []struct {
		name    string
		field   string
		want    *field.Field
		wantErr assert.ErrorAssertionFunc
	}{
		{"Simple", "a", field.String(), assert.NoError},
		{"Incorrect field", "b", nil, assert.Error},
		{"Object", "obj", field.Object("a", field.Number(field.NumberFormatFloat), "b", field.String()), assert.NoError},
		{"Object path", "obj.a", field.Number(field.NumberFormatFloat), assert.NoError},
		{"Array", "arr", field.Array(field.Object("a", field.Time())), assert.NoError},
		{"Array path", "arr.a", field.Time(), assert.NoError},
		{"Array item", "arr.", field.Object("a", field.Time()), assert.NoError},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := GetField(tt.field, sch)
			if !tt.wantErr(t, err, fmt.Sprintf("GetField(%v, sch)", tt.field)) {
				return
			}
			assert.Equalf(t, tt.want, got, "GetField(%v, sch)", tt.field)
		})
	}
}
