package items

import (
	"context"
	"fmt"
	"reflect"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/data"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	pb "git.perx.ru/perxis/perxis-go/proto/items"
	"google.golang.org/protobuf/types/known/structpb"
	"google.golang.org/protobuf/types/known/timestamppb"
)

var (
	ErrNotSystemField = errors.New("not a system field")
	ErrIncorrectValue = errors.New("incorrect value")
	ErrIncorrectField = errors.New("incorrect field")
)

type State int

func (s State) String() string {
	switch s {
	case StateDraft:
		return "Draft"
	case StateArchived:
		return "Archived"
	case StateChanged:
		return "Changed"
	case StatePublished:
		return "Published"
	}
	return "Unknown"
}

const (
	StateDraft State = iota
	StatePublished
	StateChanged
	StateArchived

	StateMax = StateArchived

	SoftDeleteSeparator = "___"
)

var PermissionsAllowAny = &Permissions{
	Edit:       true,
	Archive:    true,
	Publish:    true,
	SoftDelete: true,
	HardDelete: true,
}

// SystemFields - системные поля Item
var SystemFields = []string{
	"id",
	"space_id",
	"env_id",
	"collection_id",
	"state",
	"created_rev_at",
	"created_by",
	"created_at",
	"updated_at",
	"updated_by",
	"revision_id",
	"data",
	"translations",
	"locale",
	"deleted",
	"hidden",
	"template",
}

type Permissions struct {
	Edit       bool
	Archive    bool
	Publish    bool
	SoftDelete bool
	HardDelete bool
}

type Item struct {
	ID           string                            `json:"id" bson:"_id"` // ID - Идентификатор записи. Автоматически генерируется системой при сохранении первой ревизии.
	SpaceID      string                            `json:"spaceId" bson:"-"`
	EnvID        string                            `json:"envId" bson:"-"`
	CollectionID string                            `json:"collectionId" bson:"-"`
	State        State                             `json:"state" bson:"state"`
	CreatedRevAt time.Time                         `json:"createdRevAt,omitempty" bson:"created_rev_at,omitempty"`
	CreatedBy    string                            `json:"createdBy,omitempty" bson:"created_by,omitempty"`
	CreatedAt    time.Time                         `json:"createdAt,omitempty" bson:"created_at,omitempty"`
	UpdatedAt    time.Time                         `json:"updatedAt,omitempty" bson:"updated_at,omitempty"`
	UpdatedBy    string                            `json:"updatedBy,omitempty" bson:"updated_by,omitempty"`
	Data         map[string]interface{}            `json:"data" bson:"data"`
	Locale       string                            `json:"locale" bson:"-"`
	Translations map[string]map[string]interface{} `json:"translations" bson:"translations,omitempty"`
	RevisionID   string                            `json:"revId,omitempty" bson:"revision_id"`
	Permissions  *Permissions                      `json:"permissions,omitempty" bson:"-"`

	// Флаги записи
	Deleted  bool `json:"deleted" bson:"deleted,omitempty"`
	Hidden   bool `json:"hidden" bson:"hidden,omitempty"`
	Template bool `json:"template" bson:"template,omitempty"`
}

func NewItem(spaceID, envID, collID, id string, data map[string]interface{}, translations map[string]map[string]interface{}) *Item {
	return &Item{
		ID:           id,
		SpaceID:      spaceID,
		EnvID:        envID,
		CollectionID: collID,
		Data:         data,
		Translations: translations,
	}
}

func (i *Item) Clone() *Item {
	itm := *i
	itm.Data = data.CloneMap(i.Data)

	if i.Translations != nil {
		itm.Translations = make(map[string]map[string]interface{}, len(i.Translations))
		for t, m := range i.Translations {
			itm.Translations[t] = data.CloneMap(m)
		}
	}

	return &itm
}

func (i *Item) ToMap() map[string]interface{} {
	return map[string]interface{}{
		"id":             i.ID,
		"space_id":       i.SpaceID,
		"env_id":         i.EnvID,
		"collection_id":  i.CollectionID,
		"state":          i.State,
		"created_rev_at": i.CreatedRevAt,
		"created_by":     i.CreatedBy,
		"created_at":     i.CreatedAt,
		"updated_at":     i.UpdatedAt,
		"updated_by":     i.UpdatedBy,
		"revision_id":    i.RevisionID,
		"data":           i.Data,
		"translations":   i.Translations,
		"locale":         i.Locale,
		"deleted":        i.Deleted,
		"hidden":         i.Hidden,
		"template":       i.Template,
	}
}

func (i *Item) SetData(locale string, data map[string]interface{}) {
	if locale != "" {
		if i.Translations == nil {
			i.Translations = make(map[string]map[string]interface{})
		}
		i.Translations[locale] = data
		return
	}
	i.Data = data
}

func (i *Item) GetData(locale string) map[string]interface{} {
	if locale != "" && i.Translations != nil {
		translation, _ := i.Translations[locale]
		return MergeData(i.Data, translation)
	}
	return i.Data
}

func (i Item) Encode(ctx context.Context, s *schema.Schema) (*Item, error) {
	if i.Data != nil {
		dt, err := schema.Encode(nil, s, i.Data)
		if err != nil {
			//return errors.WithField(err, "data")
			return nil, err
		}
		i.Data = dt.(map[string]interface{})
	}
	if len(i.Translations) > 0 {
		for l, v := range i.Translations {
			dt, err := schema.Encode(nil, s, v)
			if err != nil {
				//return errors.WithField(err, fmt.Sprintf("translations.%s", l))
				return nil, err
			}
			i.Translations[l] = dt.(map[string]interface{})
		}
	}
	return &i, nil
}

func (i Item) Decode(ctx context.Context, s *schema.Schema) (res *Item, err error) {

	if i.Data != nil {
		i.Data, err = s.Decode(ctx, i.Data)
		if err != nil {
			return nil, err
			//return errors.WithField(err, "data")
		}
	}

	return &i, nil
}

// MergeData дополняет отсутствующие данные из оригинальных данных
func MergeData(data ...map[string]interface{}) map[string]interface{} {
	merge := make(map[string]interface{})
	for _, d := range data {
		for k, v := range d {
			merge[k] = v
		}
	}
	return merge
}

// ClearData убирает данные которые не изменились по сравнению с оригинальными данными
func ClearData(data ...map[string]interface{}) map[string]interface{} {
	var clear map[string]interface{}

	for _, d := range data {
		if clear == nil {
			clear = d
			continue
		}

		for k, v := range d {
			if reflect.DeepEqual(clear[k], v) {
				delete(clear, k)
			}
		}
	}

	return clear
}

type ProcessDataFunc func(ctx context.Context, sch *schema.Schema, data map[string]interface{}) (map[string]interface{}, error)

func (i Item) ProcessData(ctx context.Context, sch *schema.Schema, fn ProcessDataFunc, locales ...string) (*Item, error) {
	if i.Data != nil {
		dt, err := fn(ctx, sch, i.Data)
		if err != nil {
			return nil, errors.WithField(err, "data")
		}
		i.Data = dt
	}

	tr := make(map[string]map[string]interface{})
	for _, l := range locales {

		data := i.GetData(l)

		dt, err := fn(ctx, sch, data)
		if err != nil {
			return nil, errors.WithField(err, fmt.Sprintf("translations.%s", l))
		}
		tr[l] = dt

	}

	i.Translations = nil
	if len(tr) > 0 {
		i.Translations = tr
	}

	return &i, nil
}

// IsSystemField возвращает являться ли поле системным
func IsSystemField(field string) bool {
	if data.Contains(field, SystemFields) {
		return true
	}
	return false
}

// SetSystemField устанавливает значение системного поля
func (i *Item) SetSystemField(field string, value interface{}) error {
	ok := true
	switch field {
	case "id":
		i.ID, ok = value.(string)
	case "space_id":
		i.SpaceID, ok = value.(string)
	case "env_id":
		i.EnvID, ok = value.(string)
	case "collection_id":
		i.CollectionID, ok = value.(string)
	case "created_rev_at":
		i.CreatedRevAt, ok = value.(time.Time)
	case "created_by":
		i.CreatedBy, ok = value.(string)
	case "created_at":
		i.CreatedAt, ok = value.(time.Time)
	case "updated_by":
		i.UpdatedBy, ok = value.(string)
	case "updated_at":
		i.UpdatedAt, ok = value.(time.Time)
	case "revision_id":
		i.RevisionID, ok = value.(string)
	case "hidden":
		i.Hidden, ok = value.(bool)
	case "deleted":
		i.Deleted, ok = value.(bool)
	case "template":
		i.Template, ok = value.(bool)
	default:
		return ErrNotSystemField
	}

	if !ok {
		return ErrIncorrectValue
	}

	return nil
}

// GetSystem устанавливает значение системного поля
func (i *Item) GetSystem(field string) (any, error) {
	switch field {
	case "id":
		return i.ID, nil
	case "space_id":
		return i.SpaceID, nil
	case "env_id":
		return i.EnvID, nil
	case "collection_id":
		return i.CollectionID, nil
	case "created_rev_at":
		return i.CreatedRevAt, nil
	case "created_by":
		return i.CreatedBy, nil
	case "created_at":
		return i.CreatedAt, nil
	case "updated_by":
		return i.UpdatedBy, nil
	case "updated_at":
		return i.UpdatedAt, nil
	case "revision_id":
		return i.RevisionID, nil
	case "hidden":
		return i.Hidden, nil
	case "deleted":
		return i.Deleted, nil
	case "template":
		return i.Template, nil
	}

	return nil, ErrNotSystemField
}

func (i *Item) setItemData(field string, value interface{}) error {
	if i.Data == nil {
		i.Data = make(map[string]any)
	}

	return data.Set(field, i.Data, value)
}

func (i *Item) getItemData(field string) (any, error) {
	if i.Data != nil {
		if v, ok := data.Get(field, i.Data); ok {
			return v, nil
		}
	}

	return nil, ErrIncorrectField
}

// Set устанавливает значение поля
func (i *Item) Set(field string, value interface{}) error {
	if err := i.SetSystemField(field, value); !errors.Is(err, ErrNotSystemField) {
		return errors.Wrapf(err, "fail to set system field '%s' value", field)
	}

	return i.setItemData(field, value)
}

// Get возвращает значение поля
func (i *Item) Get(field string) (any, error) {
	if v, err := i.GetSystem(field); err == nil {
		return v, err
	}

	return i.getItemData(field)
}

// GetSystemField возвращает описание поля для системных аттрибутов Item
func GetSystemField(fld string) (*field.Field, error) {
	switch fld {
	case "id", "space_id", "env_id", "collection_id", "revision_id":
		return field.String(), nil
	case "created_rev_at", "created_at", "updated_at", "published_at":
		return field.Time(), nil
	case "created_by", "updated_by", "published_by":
		return field.String(), nil
	case "hidden", "deleted", "template":
		return field.Bool(), nil
	}

	return nil, ErrNotSystemField
}

// GetField возвращает значение поля
func GetField(field string, sch *schema.Schema) (*field.Field, error) {
	if f, err := GetSystemField(field); err == nil {
		return f, err
	}

	f := sch.GetField(field)
	if f == nil {
		return nil, ErrIncorrectField
	}

	return f, nil
}

// GetSystemNamedFields возвращает описание всех системных полей Item
func GetSystemNamedFields() []field.NamedField {
	fields := make([]field.NamedField, 0, len(SystemFields))
	for _, n := range SystemFields {
		f := field.NamedField{Name: n}
		f.Field, _ = GetSystemField(n)
		fields = append(fields, f)
	}

	return fields
}

func ItemToProto(item *Item) *pb.Item {
	if item == nil {
		return nil
	}

	protoItem := &pb.Item{
		Id:           item.ID,
		SpaceId:      item.SpaceID,
		EnvId:        item.EnvID,
		CollectionId: item.CollectionID,
		State:        pb.Item_State(item.State),
		CreatedBy:    item.CreatedBy,
		UpdatedBy:    item.UpdatedBy,
		RevisionId:   item.RevisionID,
		Locale:       item.Locale,
		Hidden:       item.Hidden,
		Template:     item.Template,
		Deleted:      item.Deleted,
	}

	if item.Data != nil {
		protoItem.Data, _ = structpb.NewStruct(item.Data)
	}
	if item.Translations != nil {
		protoItem.Translations = make(map[string]*structpb.Struct, len(item.Translations))
		for k, v := range item.Translations {
			protoItem.Translations[k], _ = structpb.NewStruct(v)
		}
	}

	protoItem.CreatedRevAt = timestamppb.New(item.CreatedRevAt)
	protoItem.CreatedAt = timestamppb.New(item.CreatedAt)
	protoItem.UpdatedAt = timestamppb.New(item.UpdatedAt)

	if item.Permissions != nil {
		protoItem.Permissions = &pb.Permissions{
			Edit:       item.Permissions.Edit,
			Archive:    item.Permissions.Archive,
			Publish:    item.Permissions.Publish,
			SoftDelete: item.Permissions.SoftDelete,
			HardDelete: item.Permissions.HardDelete,
		}
	}

	return protoItem
}

func ItemFromProto(protoItem *pb.Item) *Item {

	if protoItem == nil {
		return nil
	}

	item := &Item{
		ID:           protoItem.Id,
		SpaceID:      protoItem.SpaceId,
		EnvID:        protoItem.EnvId,
		CollectionID: protoItem.CollectionId,
		State:        State(protoItem.State),
		CreatedBy:    protoItem.CreatedBy,
		UpdatedBy:    protoItem.UpdatedBy,
		RevisionID:   protoItem.RevisionId,
		Locale:       protoItem.Locale,
		Hidden:       protoItem.Hidden,
		Template:     protoItem.Template,
		Deleted:      protoItem.Deleted,
	}

	if protoItem.Data != nil {
		item.Data = protoItem.Data.AsMap()
	}

	if protoItem.Translations != nil {
		item.Translations = make(map[string]map[string]interface{}, len(protoItem.Translations))
		for k, v := range protoItem.Translations {
			item.Translations[k] = v.AsMap()
		}
	}

	if protoItem.Permissions != nil {
		item.Permissions = &Permissions{
			Edit:       protoItem.Permissions.Edit,
			Archive:    protoItem.Permissions.Archive,
			Publish:    protoItem.Permissions.Publish,
			SoftDelete: protoItem.Permissions.SoftDelete,
			HardDelete: protoItem.Permissions.HardDelete,
		}
	}

	item.CreatedRevAt = protoItem.CreatedRevAt.AsTime()
	item.CreatedAt = protoItem.CreatedAt.AsTime()
	item.UpdatedAt = protoItem.UpdatedAt.AsTime()

	return item
}

func GetItemIDs(arr []*Item) []string {
	res := make([]string, len(arr))
	for i, e := range arr {
		res[i] = e.ID
	}
	return res
}
