package filter

import (
	"fmt"
	"reflect"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"git.perx.ru/perxis/perxis-go/pkg/schema/validate"
	"github.com/hashicorp/go-multierror"
	"github.com/mitchellh/mapstructure"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
)

type Op string

const (
	Equal          Op = "eq"
	NotEqual       Op = "neq"
	Less           Op = "lt"
	LessOrEqual    Op = "lte"
	Greater        Op = "gt"
	GreaterOrEqual Op = "gte"
	In             Op = "in"
	NotIn          Op = "nin"
	Contains       Op = "contains"
	NotContains    Op = "ncontains"
	Or             Op = "or"
	And            Op = "and"
	Near           Op = "near"
)

type Filter struct {
	Op    Op
	Field string
	Value interface{}
}

func (f Filter) Format(s fmt.State, verb rune) {
	fmt.Fprintf(s, "{Op:%s Field:%s Value:%+v}", f.Op, f.Field, f.Value)
}

func NewFilter(op Op, field string, val interface{}) *Filter {
	return &Filter{
		Op:    op,
		Field: field,
		Value: val,
	}
}

type FilterHandler struct {
	schemas  []*schema.Schema
	qbuilder QueryBuilder
	prefix   string
}

func NewFilterHandler(sch ...*schema.Schema) *FilterHandler {
	return &FilterHandler{
		schemas: sch,
		//qbuilder: qb,
	}
}

func (h *FilterHandler) SetTrimPrefix(prefix string) *FilterHandler {
	h.prefix = prefix
	return h
}

func (h *FilterHandler) removeFieldPrefix(f string) string {
	if h.prefix != "" {
		return strings.TrimPrefix(f, h.prefix+".")
	}
	return f
}

func (h *FilterHandler) AddSchema(sch ...*schema.Schema) *FilterHandler {
	for _, s := range sch {
		h.schemas = append(h.schemas, s)
	}
	return h
}

func (h *FilterHandler) SetQueryBuilder(qb QueryBuilder) {
	h.qbuilder = qb
}

func (h *FilterHandler) Validate(filter ...*Filter) (err error) {
	if len(h.schemas) == 0 {
		return errors.New("no schema provided")
	}

	for _, sch := range h.schemas {
		var merr *multierror.Error

		for _, f := range filter {
			if err := h.validate(sch, f); err != nil {
				merr = multierror.Append(merr, err)
			}
		}
		if merr != nil {
			merr.ErrorFormat = func(i []error) string {
				return fmt.Sprintf("%d validation error(s)", len(i))
			}
			return errors.WithField(merr, "filter")
		}
	}
	return nil
}

// todo: '$elemMatch' - запросы к полю-массиву на попадание в условие: '{ results: { $elemMatch: { $gte: 80, $lt: 85 } }' ?

func (h *FilterHandler) validate(sch *schema.Schema, f *Filter) (err error) {
	if f == nil {
		return
	}

	fld := h.removeFieldPrefix(f.Field)

	switch f.Op {
	case Equal, NotEqual, Less, LessOrEqual, Greater, GreaterOrEqual:
		fld := sch.GetField(fld)
		if fld == nil {
			return h.formatErr(f.Field, f.Op, errors.New("field not found in collection schema"))
		}

		if f.Value, err = schema.Decode(nil, fld, f.Value); err != nil {
			return h.formatErr(f.Field, f.Op, err)
		}
		if err = validate.Validate(nil, fld, f.Value); err != nil {
			return h.formatErr(f.Field, f.Op, err)
		}
	case In, NotIn:
		fld := sch.GetField(fld)
		if fld == nil {
			return h.formatErr(f.Field, f.Op, errors.New("field not found in collection schema"))
		}
		val := reflect.ValueOf(f.Value)
		if val.IsZero() || (val.Kind() != reflect.Array && val.Kind() != reflect.Slice) {
			return h.formatErr(f.Field, f.Op, errors.New("\"IN/NOT IN\" operations require array type for value"))
		}

		switch fld.GetType().(type) {
		case *field.ArrayType:
			f.Value, err = schema.Decode(nil, fld, f.Value)
			if err != nil {
				return h.formatErr(f.Field, f.Op, err)
			}
		default:
			decodedVal := make([]interface{}, 0, val.Len())
			for i := 0; i < val.Len(); i++ {
				v, err := schema.Decode(nil, fld, val.Index(i).Interface())
				if err != nil {
					return h.formatErr(f.Field, f.Op, err)
				}
				decodedVal = append(decodedVal, v)
			}

			f.Value = decodedVal
		}

	case Contains, NotContains:
		fld := sch.GetField(fld)
		if fld == nil {
			return h.formatErr(f.Field, f.Op, errors.New("field not found in collection schema"))
		}

		typ := fld.GetType()

		if typ.Name() != "string" && typ.Name() != "array" {
			return h.formatErr(f.Field, f.Op, errors.New("\"CONTAINS/NOT CONTAINS\" operations require field to be 'string' or 'string array'"))
		}
		if typ.Name() == "array" {
			params := fld.Params.(*field.ArrayParameters)
			if params.Item != nil || params.Item.GetType().Name() != "string" {
				return h.formatErr(f.Field, f.Op, errors.New("\"CONTAINS/NOT CONTAINS\" operations require field to be 'string' or 'string array'"))
			}
		}

		if reflect.TypeOf(f.Value).Kind() != reflect.String {
			return h.formatErr(f.Field, f.Op, errors.New("\"CONTAINS/NOT CONTAINS\" operations require value to be 'string'"))
		}

	case Or, And:
		fltrs, ok := f.Value.([]*Filter)
		if !ok {
			return h.formatErr(f.Field, f.Op, errors.New("array of filters should be provided for operations "))
		}
		for _, f := range fltrs {
			err = h.validate(sch, f)
			if err != nil {
				return err
			}
		}

	case Near:
		fld := sch.GetField(fld)
		if fld == nil {
			return h.formatErr(f.Field, f.Op, errors.New("field not found in collection schema"))
		}

		_, ok := fld.Params.(*field.LocationParameters)
		if !ok {
			return h.formatErr(f.Field, f.Op, errors.New("field must be a location"))
		}

		value, ok := f.Value.(map[string]interface{})
		if !ok {
			return h.formatErr(f.Field, f.Op, errors.New("filter value should be map"))
		}

		point, ok := value["point"]
		if !ok {
			return h.formatErr(f.Field, f.Op, errors.New("filter value should have location"))
		}

		var p field.GeoJSON
		if err := mapstructure.Decode(map[string]interface{}{"type": "Point", "coordinates": point}, &p); err != nil {
			return h.formatErr(f.Field, f.Op, err)
		}

		maxD, ok := value["distance"]
		if ok {
			v := reflect.ValueOf(maxD)
			if !v.Type().ConvertibleTo(reflect.TypeOf(float64(0))) {
				return h.formatErr(f.Field, f.Op, errors.New("filter value distance must be a number"))
			}
			val := v.Convert(reflect.TypeOf(float64(0)))
			if val.Float() < 0 {
				return h.formatErr(f.Field, f.Op, errors.New("filter value distance should not be negative"))
			}
		}

	default:
		return h.formatErr(f.Field, f.Op, errors.New("unknown operation"))
	}

	return nil
}

func (*FilterHandler) formatErr(args ...interface{}) error {
	var (
		f   string
		op  Op
		err error
	)
	for _, arg := range args {
		switch v := arg.(type) {
		case string:
			f = v
		case Op:
			op = v
		case error:
			err = v
		}
	}
	return errors.WithField(fmt.Errorf("op: '%s' %s", op, err), f)
}

func (h *FilterHandler) Query(filter ...*Filter) interface{} {
	return h.qbuilder.Query(filter...)
}

type QueryBuilder interface {
	Query(filter ...*Filter) interface{}
	SetFieldPrefix(string)
}

type mongoQueryBuilder struct {
	m      map[Op]string
	prefix string
}

func NewMongoQueryBuilder() QueryBuilder {
	b := new(mongoQueryBuilder)
	b.m = map[Op]string{
		Equal:          "$eq",
		NotEqual:       "$ne",
		Less:           "$lt",
		LessOrEqual:    "$lte",
		Greater:        "$gt",
		GreaterOrEqual: "$gte",
		In:             "$in",
		NotIn:          "$nin",
		Contains:       "$regex",
		NotContains:    "$not",
		Or:             "$or",
		And:            "$and",
		Near:           "$near",
	}
	return b
}

func (b *mongoQueryBuilder) getOp(op Op) string {
	return b.m[op]
}

func (b *mongoQueryBuilder) SetFieldPrefix(prefix string) {
	b.prefix = prefix
}

func (b *mongoQueryBuilder) Query(filters ...*Filter) interface{} {
	if len(filters) == 0 {
		return bson.M{}
	}
	filter := &Filter{Op: And, Value: filters}
	return b.query(filter)
}

func (b *mongoQueryBuilder) query(f *Filter) bson.M {
	if f == nil {
		return nil
	}

	switch f.Op {
	case Equal, NotEqual, Less, LessOrEqual, Greater, GreaterOrEqual, In, NotIn:
		return bson.M{
			b.field(f.Field): bson.M{
				b.getOp(f.Op): f.Value,
			},
		}
	case Contains, NotContains:

		val, _ := f.Value.(string)
		return bson.M{
			b.field(f.Field): bson.M{
				b.getOp(f.Op): primitive.Regex{Pattern: val},
			},
		}

	case Or, And:
		fltrs, ok := f.Value.([]*Filter)
		if !ok {
			return nil
		}

		arr := bson.A{}
		for _, fltr := range fltrs {
			arr = append(arr, b.query(fltr))
		}
		return bson.M{
			b.getOp(f.Op): arr,
		}
	case Near:
		val, ok := f.Value.(map[string]interface{})
		if ok {
			var p field.GeoJSON
			c, ok := val["point"]
			if !ok {
				return nil
			}
			if err := mapstructure.Decode(map[string]interface{}{"type": "Point", "coordinates": c}, &p); err != nil {
				return nil
			}
			q := bson.D{{Key: "$geometry", Value: p}}

			if maxD, ok := val["distance"]; ok {
				q = append(q, bson.E{Key: "$maxDistance", Value: maxD})
			}

			return bson.M{
				b.field(f.Field + ".geometry"): bson.M{b.getOp(f.Op): q},
			}
		}
	}

	return nil
}

func (b *mongoQueryBuilder) field(f string) string {
	if b.prefix == "" || strings.HasPrefix(f, b.prefix) {
		return f
	}
	return b.prefix + "." + f
}

// $text search ??
//func (b *mongoQueryBuilder) textSearchQuery(filters ...*Filter) string {
//	cnt, notcnt := "", ""
//	for _, f := range filters {
//		val, ok := f.Value.(string)
//		if !ok {
//			continue
//		}
//		switch f.Op {
//		case Contains:
//			if len(cnt) > 0 {
//				cnt += " "
//			}
//			cnt += val
//		case NotContains:
//			words := strings.Split(val, " ")
//			for _, w := range words {
//				if len(notcnt) > 0 {
//					notcnt += " "
//				}
//				notcnt += "-" + w
//			}
//		}
//	}
//	if len(cnt) == 0 {
//		return ""
//	}
//	if len(notcnt) > 0 {
//		cnt += " " + notcnt
//	}
//	return cnt
//}
