package files

import (
	"context"
	"errors"
	"fmt"
	"net/url"
	"reflect"

	"git.perx.ru/perxis/perxis-go/pkg/items"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	signer "git.perx.ru/perxis/perxis-go/pkg/urlsigner"
	"github.com/mitchellh/mapstructure"
)

const FileTypeName = "file"

type FileParameters struct {
	t *FileType
}

func (p FileParameters) Type() field.Type                   { return p.t }
func (p *FileParameters) Clone(reset bool) field.Parameters { return p }

type FileType struct {
	fs            Files
	signer        signer.URLSigner
	fileServerUrl string
	uploader      Uploader
}

func NewFileType(fs Files, signer signer.URLSigner, fileServerUrl string) *FileType {
	return &FileType{fs: fs, signer: signer, fileServerUrl: fileServerUrl, uploader: NewUploader()}
}

func (t *FileType) WithUploader(uploader Uploader) *FileType {
	t.uploader = uploader
	return t
}

func (t FileType) Name() string { return FileTypeName }

func (t *FileType) NewParameters() field.Parameters {
	return &FileParameters{t}
}

func (t FileType) Decode(_ context.Context, fld *field.Field, v interface{}) (interface{}, error) {
	if v == nil {
		return nil, nil
	}
	var f File
	if err := mapstructure.Decode(v, &f); err != nil {
		return nil, err
	}
	return &f, nil
}

func (t FileType) Encode(ctx context.Context, fld *field.Field, v interface{}) (interface{}, error) {
	if v == nil {
		return nil, nil
	}

	f, ok := v.(*File)
	if !ok {
		return nil, fmt.Errorf("FileField encode error: incorrect type: \"%s\", expected \"file\"", reflect.ValueOf(v).Kind())
	}

	if f.File != nil { // upload from file system
		upload, err := t.fs.Upload(ctx, f)
		if err != nil {
			return nil, err
		}
		if err = t.uploader.Upload(f.File, upload); err != nil {
			return nil, err
		}
		f = &upload.File
	}

	u := fmt.Sprintf("%s/%s", t.fileServerUrl, f.ID)
	resURL, err := url.Parse(u)
	if err != nil {
		return nil, err
	}
	if t.signer != nil {
		f.URL = t.signer.Sign(resURL).String()
	} else {
		f.URL = resURL.String()
	}
	res := make(map[string]interface{})
	if err := mapstructure.Decode(f, &res); err != nil {
		return nil, err
	}

	return res, nil

}

// PreSave - функция буде вызвана перед сохранением поля в Storage. Реализует интерфейс `perxis.PreSaver`
// Выполняет проверку поля является ли файл только что загруженным и переносит его при необходимости для
// постоянного хранения
func (t FileType) PreSave(ctx context.Context, fld *field.Field, v interface{}, itemCtx *items.Context) (interface{}, bool, error) {
	if v == nil {
		return nil, false, nil
	}
	f := v.(*File)
	if f.ID == "" {
		return nil, false, errors.New("FileType: file id required")
	}
	if !f.Temporary() {
		return f, false, nil
	}

	f, err := t.fs.MoveUpload(ctx, &MultipartUpload{File: *f})
	if err != nil {
		return nil, false, err
	}
	return f, true, nil
}

// Field - создает новое поле Field типа FileType
// FileType должен быть предварительно создан `NewFileType` и зарегистрирован `field.Register`
func Field(o ...interface{}) *field.Field {
	t, ok := field.GetType(FileTypeName)
	if !ok {
		panic("field file type not registered")
	}
	return field.NewField(t.NewParameters(), o...)
}

func (t *FileType) IsEmpty(v interface{}) bool {
	if v == nil {
		return true
	}

	f, ok := v.(*File)

	return !ok || f.ID == ""
}

func (p FileParameters) GetField(path string) (fld *field.Field) {
	switch path {
	case "id", "name", "mimeType", "url", "key":
		return field.String()
	case "size":
		return field.Number(field.NumberFormatInt)
	default:
		return nil
	}
}

func init() {
	// По умолчанию без FS
	// Если нужны подписанные URL, и загрузка на FS, нужно зарегистрировать корректный типа
	// См. cmd/content/command/server.go:195
	field.Register(NewFileType(nil, nil, ""))
}
