package files

import (
	"encoding/base64"
	"errors"
	"io"
	"net/http"
)

type Downloader interface {
	Download(dst io.Writer, file *File) error
}

type downloader struct{}

func NewDownloader() Downloader {
	return &downloader{}
}

func (d *downloader) Download(dst io.Writer, file *File) error {
	r, err := http.Get(file.URL)
	if err != nil {
		return err
	}
	defer r.Body.Close()

	if r.StatusCode != http.StatusOK {
		return errors.New("download request failed: " + r.Status)
	}

	_, err = io.Copy(dst, r.Body)
	return err
}

type noopDownloader struct{}

func NewNoopDownloader() Downloader {
	return &noopDownloader{}
}

func (d *noopDownloader) Download(dst io.Writer, file *File) error {
	return nil
}

type dummyDownloader struct{}

func NewDummyDownloader() Downloader {
	return &dummyDownloader{}
}

func (d *dummyDownloader) Download(dst io.Writer, file *File) error {
	// png pixel 10x10
	pixel, err := base64.StdEncoding.DecodeString("iVBORw0KGgoAAAANSUhEUgAAAAoAAAAKCAYAAACNMs+9AAAAFUlEQVR42mNkYPhfz0AEYBxVSF+FAP5FDvcfRYWgAAAAAElFTkSuQmCC")
	_, err = dst.Write(pixel)
	return err
}
