package extension

import (
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetErrors(t *testing.T) {

	getOut := func(results ...*RequestResult) results { return &InstallResponse{Results: results} }

	tests := []struct {
		name     string
		out      results
		err      error
		wrapErr  error
		checkErr func(t *testing.T, err error)
	}{
		{
			name: "no errors",
			out:  getOut(&RequestResult{State: RequestOK, Extension: "a"}),
		},
		{
			name:    "no error",
			out:     getOut(&RequestResult{State: RequestOK, Extension: "a"}),
			wrapErr: ErrInstall,
		},
		{
			name:    "one errored result",
			out:     getOut(&RequestResult{State: RequestError, Extension: "a", Error: "some err", Msg: "Ошибка"}),
			wrapErr: ErrInstall,
			checkErr: func(t *testing.T, err error) {
				assert.ErrorIs(t, err, ErrInstall)
				errs := errors.GetErrors(err)
				require.Len(t, errs, 1)

				extErr := errs[0]
				assert.Equal(t, "a", ExtensionFromError(extErr))
				assert.Equal(t, "some err", extErr.Error())
				assert.Equal(t, "Ошибка", errors.GetDetail(extErr))
			},
		},
		{
			name: "multiple results, some of them errored",
			out: getOut(
				&RequestResult{State: RequestError, Extension: "a", Error: "some err a", Msg: "Ошибка А"},
				&RequestResult{State: RequestOK, Extension: "b"},
				&RequestResult{State: RequestError, Extension: "c", Error: "some err c", Msg: "Ошибка С"},
				&RequestResult{State: RequestOK, Extension: "d"},
			),
			wrapErr: ErrInstall,
			checkErr: func(t *testing.T, err error) {
				assert.ErrorIs(t, err, ErrInstall)
				errs := errors.GetErrors(err)
				require.Len(t, errs, 2)

				extErr1 := errs[0]
				assert.Equal(t, "a", ExtensionFromError(extErr1))
				assert.Equal(t, "some err a", extErr1.Error())
				assert.Equal(t, "Ошибка А", errors.GetDetail(extErr1))

				extErr2 := errs[1]
				assert.Equal(t, "c", ExtensionFromError(extErr2))
				assert.Equal(t, "some err c", extErr2.Error())
				assert.Equal(t, "Ошибка С", errors.GetDetail(extErr2))
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			err := getErrors(tt.out, tt.err, tt.wrapErr)
			if tt.checkErr != nil {
				require.Error(t, err)
				tt.checkErr(t, err)
			} else {
				require.NoError(t, err)
			}
		})
	}
}
