package expr

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"golang.org/x/net/context"
)

func TestFormat(t *testing.T) {
	ctx := context.Background()

	tests := []struct {
		name  string
		eval  string
		env   map[string]interface{}
		wantB interface{}
	}{
		{"sprintf#1", "sprintf(s1, s2)", map[string]interface{}{"s1": "hello %s", "s2": "world"}, "hello world"},
		{"sprintf#2", "sprintf(s1, s2)", map[string]interface{}{"s1": nil, "s2": nil}, ""},
		{"sprintf#3", "sprintf(s1, s2)", map[string]interface{}{"s1": "hello %s", "s2": nil}, "hello %!s(<nil>)"},
		{"to_upper#1", "to_upper(s1)", map[string]interface{}{"s1": "hello"}, "HELLO"},
		{"to_upper#2", "to_upper(s1)", map[string]interface{}{"s1": nil}, ""},
		{"trim_space#1", "trim_space(s1)", map[string]interface{}{"s1": " hel lo  wor ld  "}, "hel lo  wor ld"},
		{"trim_space#2", "trim_space(s1)", map[string]interface{}{"s1": nil}, ""},
		{"slugify#1", "slugify(s1)", map[string]interface{}{"s1": " hello  world "}, "hello-world"},
		{"slugify#2", "slugify(s1)", map[string]interface{}{"s1": " привет  мир "}, "privet-mir"},
		{"slugify#3", "slugify(s1)", map[string]interface{}{"s1": "privet-mir"}, "privet-mir"},
		{"slugify#4", "slugify(s1)", nil, ""},
		{"make_path#1", "make_path(s1, s2)", map[string]interface{}{"s1": "/pages", "s2": "page-1"}, "/pages/page-1"},
		{"make_path#2", "make_path(s1, s2)", map[string]interface{}{"s1": "/pages", "s2": ""}, "/pages"},
		{"make_path#3", "make_path(s1, s2)", map[string]interface{}{"s1": "", "s2": "pages"}, "/pages"},
		{"make_path#4", "make_path(s1, s2)", map[string]interface{}{"s1": "", "s2": ""}, "/"},
		{"make_path#4", "make_path(s1, s2)", nil, "/"},
		{"make_path#5", "make_path(s1, slugify(s2))", map[string]interface{}{"s1": "/pages", "s2": " привет  мир "}, "/pages/privet-mir"},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			gotB, err := Eval(ctx, tt.eval, tt.env)
			require.NoError(t, err)
			assert.Equal(t, tt.wantB, gotB)
		})
	}
}
