//go:build integration

package events

import (
	"testing"
	"time"

	pb "git.perx.ru/perxis/perxis-go/pkg/events/test_proto"
	"github.com/golang/protobuf/proto"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type Test struct {
	Text string
}

func (t *Test) ToProto() (proto.Message, error) {
	return &pb.Test{Text: t.Text}, nil
}

func (t *Test) FromProto(message proto.Message) error {
	t.Text = message.(*pb.Test).Text
	return nil
}

func TestNatsBroker(t *testing.T) {

	b, err := Open("nats://localhost:4222", "")
	require.NoError(t, err)

	resCh := make(chan string, 3)
	_, err = b.Subscribe("a.*.c.>", func(t *Test) { resCh <- t.Text })
	require.NoError(t, err)

	require.NoError(t, b.Publish("a.b.c", &Test{Text: "1"}))
	require.NoError(t, b.Publish("a.b.c.d", &Test{Text: "2"}))
	require.NoError(t, b.Publish("a.b.c.d.e", &Test{Text: "3"}))
	require.NoError(t, b.Publish("a.x.c", &Test{Text: "4"}))
	require.NoError(t, b.Publish("a.x.c.d", &Test{Text: "5"}))

	time.Sleep(200 * time.Millisecond)
	require.NoError(t, b.Close())
	close(resCh)
	assert.ElementsMatch(t, []string{"2", "3", "5"}, func() []string {
		var res []string
		for v := range resCh {
			res = append(res, v)
		}
		return res
	}())
}

func TestTags(t *testing.T) {

	b, err := Open("nats://localhost:4222", "")
	require.NoError(t, err)

	resCh := make(chan string, 3)
	_, err = b.Subscribe("a.*.c.>", func(t *Test) { resCh <- t.Text }, FilterTag("one", "two", "three"))
	require.NoError(t, err)

	require.NoError(t, b.Publish("a.b.c", &Test{Text: "1"}))
	require.NoError(t, b.Publish("a.b.c.d", &Test{Text: "2"}))
	require.NoError(t, b.Publish("a.b.c.d.e", &Test{Text: "3"}, Tag("one")))
	require.NoError(t, b.Publish("a.x.c", &Test{Text: "4"}))
	require.NoError(t, b.Publish("a.x.c.d", &Test{Text: "5"}, Tag("two")))
	require.NoError(t, b.Publish("a.x.c.d", &Test{Text: "6"}, Tag("two", "one")))
	require.NoError(t, b.Publish("a.x.c.d", &Test{Text: "7"}, Tag("four")))

	time.Sleep(200 * time.Millisecond)
	require.NoError(t, b.Close())
	close(resCh)
	assert.ElementsMatch(t, []string{"3", "5", "6"}, func() []string {
		var res []string
		for v := range resCh {
			res = append(res, v)
		}
		return res
	}())
}
