package errors

import (
	"fmt"
	"io"
)

type ErrorSource struct {
	Pointer   string `json:"pointer,omitempty"`
	Parameter string `json:"parameter,omitempty"`
	Field     string `json:"field,omitempty"`
}

func NewSource(args ...string) *ErrorSource {
	s := &ErrorSource{}
	if len(args) > 0 {
		s.Pointer = args[0]
	}
	if len(args) > 1 {
		s.Parameter = args[1]
	}
	if len(args) > 2 {
		s.Field = args[2]
	}
	return s
}

type ErrSource interface{ Source() *ErrorSource }

type withSource struct {
	err    error
	source *ErrorSource
}

func (w *withSource) Source() *ErrorSource { return w.source }
func (w *withSource) Error() string        { return w.err.Error() }
func (w *withSource) Cause() error         { return w.err }
func (w *withSource) Unwrap() error        { return w.err }

func (w *withSource) Format(s fmt.State, verb rune) {
	switch verb {
	case 'v':
		if s.Flag('+') {
			fmt.Fprintf(s, "%+v", w.Cause())
			fmt.Fprintf(s, "\nerror source: %+v", w.source)
			return
		}
		fallthrough
	case 's':
		io.WriteString(s, w.Error())
	case 'q':
		fmt.Fprintf(s, "%q", w.Error())
	}
}

func WithSource(err error, source *ErrorSource) error {
	if err == nil {
		return nil
	}
	return &withSource{err: err, source: source}
}

func GetSource(err error) *ErrorSource {
	for e := err; e != nil; e = Unwrap(e) {
		if errSource, ok := e.(ErrSource); ok {
			return errSource.Source()
		}
	}
	return nil
}
