package errors

import (
	"fmt"
	"io"

	"github.com/rs/xid"
)

type ErrID interface{ ID() string }

type withId struct {
	err error
	id  string
}

func (w *withId) ID() string    { return w.id }
func (w *withId) Error() string { return w.err.Error() }
func (w *withId) Cause() error  { return w.err }
func (w *withId) Unwrap() error { return w.err }

func (w *withId) Format(s fmt.State, verb rune) {
	switch verb {
	case 'v':
		if s.Flag('+') {
			fmt.Fprintf(s, "%+v", w.Cause())
			fmt.Fprintf(s, "\nerror id: %s", w.id)
			return
		}
		fallthrough
	case 's':
		io.WriteString(s, w.Error())
	case 'q':
		fmt.Fprintf(s, "%q", w.Error())
	}
}

func SetID(err error, id string) error {
	if GetID(err) != "" {
		return err
	}
	return &withId{err: err, id: id}
}

func GetID(err error) string {
	for e := err; e != nil; e = Unwrap(e) {
		if errID, ok := e.(ErrID); ok {
			return errID.ID()
		}
	}
	return ""
}

func WithID(err error) error {
	if err == nil {
		return nil
	}
	id := GetID(err)
	if id == "" {
		id = xid.New().String()
		err = &withId{err: err, id: id}
	}
	return err
}
