package grpc

import (
	"reflect"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/proto/common"
	"github.com/hashicorp/go-multierror"
)

func Test_getBadRequest(t *testing.T) {
	tests := []struct {
		name string
		err  error
		want *common.Error_BadRequest
	}{
		{"Without field error", errors.New("some"), nil},
		{"Field error", errors.WithField(errors.New("some"), "fld1"),
			&common.Error_BadRequest{Errors: []*common.Error_BadRequest_FieldViolation{{Description: "some", Field: "fld1"}}}},
		{"Deep field error", errors.Wrap(errors.WithField(errors.New("some"), "fld1"), "extra"),
			&common.Error_BadRequest{Errors: []*common.Error_BadRequest_FieldViolation{{Description: "some", Field: "fld1"}}}},
		{"Nested field errors", errors.WithField(errors.WithField(errors.New("some"), "fld1"), "fld2"),
			&common.Error_BadRequest{Errors: []*common.Error_BadRequest_FieldViolation{{Description: "some", Field: "fld2.fld1"}}}},
		{"Deep nested field errors", errors.WithField(errors.Wrap(errors.WithField(errors.New("some"), "fld1"), "extra"), "fld2"),
			&common.Error_BadRequest{Errors: []*common.Error_BadRequest_FieldViolation{{Description: "some", Field: "fld2.fld1"}}}},
		{"Multi Field error", multierror.Append(errors.WithField(errors.New("some"), "fld1"), errors.WithField(errors.New("some"), "fld2")),
			&common.Error_BadRequest{Errors: []*common.Error_BadRequest_FieldViolation{{Description: "some", Field: "fld1"}, {Description: "some", Field: "fld2"}}}},
		{"Nested Multi Field error", errors.WithField(multierror.Append(errors.WithField(errors.New("some"), "fld1"), errors.WithField(errors.New("some"), "fld2")), "fld3"),
			&common.Error_BadRequest{Errors: []*common.Error_BadRequest_FieldViolation{{Description: "some", Field: "fld3.fld1"}, {Description: "some", Field: "fld3.fld2"}}}},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := getBadRequest(tt.err); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("getBadRequest() = %v, want %v", got, tt.want)
			}
		})
	}
}

func Test_getHelp(t *testing.T) {
	tests := []struct {
		name string
		err  error
		want *common.Error_Help
	}{
		{"Without help", errors.New("some"), nil},
		{"With help",
			errors.WithHelp(errors.New("some"), &errors.Help{Links: []errors.HelpLink{{Description: "Help1", URL: "Url1"}}}),
			&common.Error_Help{Links: []*common.Error_Help_Link{{Description: "Help1", Url: "Url1"}}}},
		{"With nested help",
			errors.WithHelp(errors.WithHelp(errors.New("some"), &errors.Help{Links: []errors.HelpLink{{Description: "Help2", URL: "Url2"}}}), &errors.Help{Links: []errors.HelpLink{{Description: "Help1", URL: "Url1"}}}),
			&common.Error_Help{Links: []*common.Error_Help_Link{{Description: "Help1", Url: "Url1"}, {Description: "Help2", Url: "Url2"}}}},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := getHelp(tt.err); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("getHelp() = %v, want %v", got, tt.want)
			}
		})
	}
}
