package grpc

import (
	"fmt"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/proto/common"
	"github.com/hashicorp/go-multierror"
	"google.golang.org/grpc/status"
)

func getHelp(err error) *common.Error_Help {
	if help := errors.GetHelp(err); help != nil {
		pbhelp := &common.Error_Help{}
		for _, hl := range help.Links {
			pbhelp.Links = append(pbhelp.Links, &common.Error_Help_Link{Description: hl.Description, Url: hl.URL})
		}
		return pbhelp
	}
	return nil
}

func helpFromPb(err error, pberr *common.Error) error {
	if pberr.Help == nil || len(pberr.Help.Links) == 0 {
		return err
	}
	h := &errors.Help{}
	for _, l := range pberr.Help.Links {
		h.Links = append(h.Links, errors.HelpLink{Description: l.Description, URL: l.Url})
	}
	return errors.WithHelp(err, h)
}

func getBadRequest(err error) *common.Error_BadRequest {
	br := &common.Error_BadRequest{}

	var merr *multierror.Error
	if errors.As(err, &merr) {
		for _, e := range merr.Errors {
			var errField errors.FieldError
			if errors.As(e, &errField) {
				br.Errors = append(br.Errors, &common.Error_BadRequest_FieldViolation{Field: errField.Field(), Description: errors.Unwrap(errField).Error()})
			}
		}
	} else {
		var errField errors.FieldError
		if errors.As(err, &errField) {
			br.Errors = append(br.Errors, &common.Error_BadRequest_FieldViolation{Field: errField.Field(), Description: errors.Unwrap(errField).Error()})
		}
	}

	if len(br.Errors) > 0 {
		return br
	}

	return nil
}

func getDebugInfo(err error) *common.Error_DebugInfo {
	if trace, ok := errors.GetStackTrace(err); ok {
		di := &common.Error_DebugInfo{}
		for _, t := range trace {
			di.StackTrace = append(di.StackTrace, fmt.Sprintf("%+v", t))
		}
		return di
	}

	return nil
}

func badRequestFromPb(err error, pberr *common.Error) error {
	if pberr.BadRequest == nil || len(pberr.BadRequest.Errors) == 0 {
		return err
	}
	var merr error
	for _, e := range pberr.BadRequest.Errors {
		merr = multierror.Append(merr, errors.WithField(errors.New(e.Description), e.Field))
	}
	return errors.WrapErr(err, merr)
}

func StatusFromError(err error) *status.Status {
	if err == nil {
		return nil
	}
	err = errors.WithID(err)
	pberr := &common.Error{
		ErrorCode:         errors.GetCode(err),
		ErrorId:           errors.GetID(err),
		Reason:            "", // TBD
		Domain:            errors.GetDomain(err),
		Help:              getHelp(err),
		Metadata:          nil, // TBD
		BadRequest:        getBadRequest(err),
		DebugInfo:         getDebugInfo(err),
		LocalizedMessages: nil, // TBD
	}

	st := status.New(errors.GetStatusCode(err), err.Error())
	st, _ = st.WithDetails(pberr)
	return st
}

func convertProtoError(err error, pberr *common.Error) error {
	if pberr == nil {
		return nil
	}

	// Должен быть первым в цепочке
	err = badRequestFromPb(err, pberr)

	if pberr.ErrorCode != 0 {
		err = errors.WithCode(err, pberr.ErrorCode)
	}

	if pberr.ErrorId != "" {
		err = errors.SetID(err, pberr.ErrorId)
	}

	if pberr.Domain != "" {
		err = errors.WithDomain(err, pberr.Domain)
	}

	err = helpFromPb(err, pberr)
	return err
}

func ErrorFromStatus(st *status.Status) error {
	details := st.Details()
	err := errors.New(st.Message())
	if len(details) > 0 {
		if pberr, ok := details[0].(*common.Error); ok {
			err = convertProtoError(err, pberr)
		}
	}
	err = errors.WithStatusCode(err, st.Code())
	return err
}
