package errors

import (
	"fmt"
	"io"
)

type ErrDetail interface{ Detail() string }

type withDetail struct {
	err    error
	detail string
}

func (w *withDetail) Detail() string { return w.detail }
func (w *withDetail) Error() string  { return w.err.Error() }
func (w *withDetail) Cause() error   { return w.err }
func (w *withDetail) Unwrap() error  { return w.err }

func (w *withDetail) Format(s fmt.State, verb rune) {
	switch verb {
	case 'v':
		if s.Flag('+') {
			fmt.Fprintf(s, "%+v", w.Cause())
			fmt.Fprintf(s, "\nerror detail: %s", w.detail)
			return
		}
		fallthrough
	case 's':
		io.WriteString(s, w.Error())
	case 'q':
		fmt.Fprintf(s, "%q", w.Error())
	}
}

func WithDetail(err error, detail string) error {
	if err == nil {
		return nil
	}
	return &withDetail{err: err, detail: detail}
}

func WithDetailf(err error, format string, args ...interface{}) error {
	if err == nil {
		return nil
	}
	return &withDetail{err: err, detail: fmt.Sprintf(format, args...)}
}

func GetDetail(err error) string {
	for e := err; e != nil; e = Unwrap(e) {
		if errDetail, ok := e.(ErrDetail); ok {
			return errDetail.Detail()
		}
	}
	return ""
}
