package errors

import (
	"fmt"
	"io"
	"sync"
)

var codeErrors sync.Map

type CodeError interface {
	Code() uint64
}

type codeError struct {
	code uint64
	err  error
}

func (e *codeError) Error() string { return fmt.Sprintf("(%d) %s", e.code, e.err.Error()) }
func (e *codeError) Code() uint64  { return e.code }
func (w *codeError) Unwrap() error { return w.err }

func (e *codeError) Format(s fmt.State, verb rune) {
	switch verb {
	case 'v':
		if s.Flag('+') {
			fmt.Fprintf(s, "\n(%d): %s", e.code, e.Error())
			return
		}
		fallthrough
	case 's':
		io.WriteString(s, e.Error())
	case 'q':
		fmt.Fprintf(s, "%q", e.Error())
	}
}

func (e *codeError) Is(err error) bool {
	if ce, ok := err.(CodeError); ok {
		return e.code == ce.Code()
	}
	return false
}

func WithCode(err error, code uint64) error {
	err = &codeError{code: code, err: err}
	v, _ := codeErrors.LoadOrStore(code, err)
	return v.(error)
}

func GetCode(err error) uint64 {
	for e := err; e != nil; e = Unwrap(e) {
		if errCode, ok := e.(CodeError); ok {
			return errCode.Code()
		}
	}
	return 0
}
