package service

import (
	"context"

	"git.perx.ru/perxis/perxis-go/pkg/collections"
	"git.perx.ru/perxis/perxis-go/pkg/delivery"
	"git.perx.ru/perxis/perxis-go/pkg/environments"
	"git.perx.ru/perxis/perxis-go/pkg/items"
	"git.perx.ru/perxis/perxis-go/pkg/locales"
)

type Config struct {
	locales.Locales
	environments.Environments
	collections.Collections
	items.Items
}

func NewService(config *Config) delivery.Delivery {
	return &deliveryService{
		Locales:      config.Locales,
		Environments: config.Environments,
		Collections:  config.Collections,
		Items:        config.Items,
	}
}

type deliveryService struct {
	locales.Locales
	environments.Environments
	collections.Collections
	items.Items
}

func (s deliveryService) ListLocales(ctx context.Context, spaceId string) (locales []*locales.Locale, err error) {
	return s.Locales.List(ctx, spaceId)
}

func (s deliveryService) GetEnvironment(ctx context.Context, spaceId, envId string) (env *environments.Environment, err error) {
	return s.Environments.Get(ctx, spaceId, envId)
}

func (s deliveryService) ListEnvironments(ctx context.Context, spaceId string) (envs []*environments.Environment, err error) {
	return s.Environments.List(ctx, spaceId)
}

func (s deliveryService) GetCollection(ctx context.Context, spaceId, envId, collectionId string) (collection *collections.Collection, err error) {
	// Для Delivery всегда загружается коллекция для последующего использования, так как нет кейсов использования незагруженной коллекции
	return s.Collections.Get(ctx, spaceId, envId, collectionId)
}

func (s deliveryService) ListCollections(ctx context.Context, spaceId, envId string) (collections []*collections.Collection, err error) {
	return s.Collections.List(ctx, spaceId, envId, nil)
}

func (s deliveryService) GetItem(ctx context.Context, spaceId, envId, collectionId, itemId string, options ...*items.GetPublishedOptions) (item *items.Item, err error) {
	return s.Items.GetPublished(ctx, spaceId, envId, collectionId, itemId, options...)
}

func (s deliveryService) FindItems(ctx context.Context, spaceId, envId, collectionId string, filter *items.Filter, options ...*items.FindPublishedOptions) (items []*items.Item, total int, err error) {
	return s.Items.FindPublished(ctx, spaceId, envId, collectionId, filter, options...)
}

func (s deliveryService) Aggregate(ctx context.Context, spaceId, envId, collectionId string, filter *items.Filter, options ...*items.AggregatePublishedOptions) (result map[string]interface{}, err error) {
	return s.Items.AggregatePublished(ctx, spaceId, envId, collectionId, filter, options...)
}
