package client

import (
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	collections "git.perx.ru/perxis/perxis-go/pkg/collections/middleware"
	"git.perx.ru/perxis/perxis-go/pkg/delivery"
	deliveryservice "git.perx.ru/perxis/perxis-go/pkg/delivery/service"
	deliverytransportgrpc "git.perx.ru/perxis/perxis-go/pkg/delivery/transport/grpc"
	environments "git.perx.ru/perxis/perxis-go/pkg/environments/middleware"
	items "git.perx.ru/perxis/perxis-go/pkg/items/middleware"
	locales "git.perx.ru/perxis/perxis-go/pkg/locales/middleware"
	"google.golang.org/grpc"
)

func NewClient(conn *grpc.ClientConn, opts ...Option) (delivery.Delivery, error) {

	c := new(Config)
	for _, o := range opts {
		o(&c.Config)
	}

	client := deliverytransportgrpc.NewGRPCClient(conn, "", c.ClientOptions...)

	cfg := &deliveryservice.Config{
		Locales:      localesAdapter{dv: client},
		Environments: envsAdapter{dv: client},
		Collections:  collectionsAdapter{dv: client},
		Items:        itemsAdapter{dv: client},
	}

	if !c.NoDecode {
		cfg.Items = items.ClientEncodeMiddleware(cfg.Collections)(cfg.Items)
	}

	if !c.NoCache {
		cfg = WithCaching(cfg, 0, 0)
	}

	return deliveryservice.NewService(cfg), nil
}

func WithCaching(cfg *deliveryservice.Config, size int, ttl time.Duration) *deliveryservice.Config {
	c := *cfg

	c.Environments = environments.CachingMiddleware(cache.NewCache(size, ttl))(cfg.Environments)
	c.Locales = locales.CachingMiddleware(cache.NewCache(size, ttl))(cfg.Locales)
	c.Items = items.CachingMiddleware(cache.NewCache(size, ttl), cache.NewCache(size, ttl), c.Environments)(cfg.Items)
	c.Collections = collections.CachingMiddleware(cache.NewCache(size, ttl), c.Environments)(cfg.Collections)

	return &c
}
