package data

import (
	"fmt"
	"reflect"
	"testing"
)

func TestIsRegExp(t *testing.T) {
	tests := []struct {
		str  string
		want bool
	}{
		{"^some", true},
		{"some$", true},
		{"some.*", true},
		{"some.+", true},
		{"some*", false},
		{"some+", false},
		{"so.me*", false},
		{"some.", false},
		{"s^om$e", false},
	}
	for i, tt := range tests {
		t.Run(fmt.Sprintf("test%d", i), func(t *testing.T) {
			if got := IsRegExp(tt.str); got != tt.want {
				t.Errorf("IsRegExp(\"%s\") = %v, want %v", tt.str, got, tt.want)
			}
		})
	}
}

func TestGlobToRegexp(t *testing.T) {
	tests := []struct {
		name string
		args string
		want interface{}
	}{
		{"from start", "aaa*", "^aaa"},
		{"from end", "*aaa", "aaa$"},
		{"middle", "*aaa*", "aaa"},
		{"start end", "aa*bb", "^aa.*bb$"},
		{"middle and chars", "aa*bb??", "^aa.*bb..$"},
		{"anything", "*", "*"},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := GlobToRegexp(tt.args); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Match() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestGlobMatch(t *testing.T) {
	tests := []struct {
		name string
		s    string
		glob []string
		want bool
	}{
		{"from start", "aaateststring", []string{"aaa*"}, true},
		{"from end", "teststringaaa", []string{"*aaa"}, true},
		{"middle", "testaaastring", []string{"*aaa*"}, true},
		{"start end", "aateststringbb", []string{"aa*bb"}, true},
		{"any of", "teststringaa", []string{"aa*", "aa*bb", "*aa"}, true},
		{"none matches", "teststringaa", []string{"aa*", "aa*bb", "*bb"}, false},
		{"middle and chars", "aateststringbbcc", []string{"aa*bb??"}, true},
		{"anything", "teststring", []string{"*"}, true},
		{"false from start", "aateststring", []string{"aaa*"}, false},
		{"false from end", "teststringaa", []string{"*aaa"}, false},
		{"false middle", "testaastring", []string{"*aaa*"}, false},
		{"false start end", "aateststringb", []string{"aa*bb"}, false},
		{"false middle and chars", "ateststringbbcc", []string{"aa*bb??"}, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := GlobMatch(tt.s, tt.glob...); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Match() = %v, want %v", got, tt.want)
			}
		})
	}
}
