package data

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestMergeYaml(t *testing.T) {
	tests := []struct {
		name     string
		source   string
		override string
		res      string
		wantErr  bool
	}{
		{"simple change",
			"a: 1\nb: cat",
			"a: 2",
			"a: 2\nb: cat\n",
			false,
		},
		{"simple not change",
			"a: 1\nb: cat",
			"a: 1",
			"a: 1\nb: cat\n",
			false,
		},
		{"object merge",
			`
keyA1: val1
keyA2: val2
keyB1: val1
keyB2: val2`,
			`
keyС1: val1
keyС2: val2
keyВ1: val1
keyВ2: val2`,
			`keyA1: val1
keyA2: val2
keyB1: val1
keyB2: val2
keyВ1: val1
keyВ2: val2
keyС1: val1
keyС2: val2
`,
			false,
		},
		{"object change",
			`
keyA1: val1
keyA2: val2
keyB1: val1
keyB2: val2`,
			`
keyA1: val10
keyA2: val20
keyB1: val10
keyB2: val20`,
			`keyA1: val10
keyA2: val20
keyB1: val10
keyB2: val20
`,
			false,
		},
		{"slice of objects merge",
			`entries: 
  - keyA1: val1
    keyA2: val2
  - keyB1: val1
    keyB2: val2
  - val3`,
			`entries: 
  - keyC1: val1
    keyC2: val2
  - keyD1: val1
    keyD2: val2
  - val30`,
			`entries:
    - keyA1: val1
      keyA2: val2
    - keyB1: val1
      keyB2: val2
    - val3
    - keyC1: val1
      keyC2: val2
    - keyD1: val1
      keyD2: val2
    - val30
`,
			false,
		},
		{"slice of objects change",
			`entries: 
  - keyA1: val1
    keyA2: val2
  - keyB1: val1
    keyB2: val2
  - val3`,
			`entries:
  - keyA1: val10
    keyA2: val20
  - keyB1: val10
    keyB2: val20`,
			`entries:
    - keyA1: val1
      keyA2: val2
    - keyB1: val1
      keyB2: val2
    - val3
    - keyA1: val10
      keyA2: val20
    - keyB1: val10
      keyB2: val20
`,
			false,
		},
		{"invalid string (missing space)",
			"a:1",
			"a:2",
			"",
			true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			res, err := MergeYaml(tt.source, tt.override)
			assert.Equal(t, tt.res, res)
			if tt.wantErr {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
		})
	}
}
