package data

import (
	"crypto/rand"
	"crypto/sha512"
	"encoding/base64"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"golang.org/x/crypto/bcrypt"
)

func GetHash(value []byte) ([]byte, error) {
	return bcrypt.GenerateFromPassword(value, 0)
}

func CompareHash(value, hash []byte) error {
	return bcrypt.CompareHashAndPassword(hash, value)
}

func GetHashString(value string) (string, error) {
	hash, err := GetHash([]byte(value))
	if err != nil {
		return "", err
	}
	return string(hash), nil
}

func CompareHashString(value, hash string) error {
	return CompareHash([]byte(value), []byte(hash))
}

func GenerateRandomBytes(n int) []byte {
	b := make([]byte, n)
	_, err := rand.Read(b)
	// Note that err == nil only if we read len(b) bytes.
	if err != nil {
		panic(errors.Wrap(err, "rand.Read error"))
	}

	return b
}

func GenerateRandomString(n int) string {
	const letters = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz-"
	bytes := GenerateRandomBytes(n)

	for i, b := range bytes {
		bytes[i] = letters[b%byte(len(letters))]
	}
	return string(bytes)
}

func GenerateRandomStringURLSafe(n int) string {
	b := GenerateRandomBytes(n)
	return base64.URLEncoding.EncodeToString(b)
}

func SignValue(signature, value string) string {
	hash := sha512.Sum512([]byte(value + "_" + signature))
	encoded := base64.StdEncoding.EncodeToString(hash[:])

	return encoded
}
