package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/collaborators"
	csmocks "git.perx.ru/perxis/perxis-go/pkg/collaborators/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestCollaboratorsCache(t *testing.T) {

	const (
		userID    = "userID"
		spaceID   = "spaceID"
		spaceRole = "spaceRole"
		size      = 5
		ttl       = 20 * time.Millisecond
	)

	errNotFound := errors.NotFound(errors.New("not found"))

	ctx := context.Background()

	t.Run("Get from cache", func(t *testing.T) {
		cs := &csmocks.Collaborators{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

		cs.On("Get", mock.Anything, spaceID, userID).Return(spaceRole, nil).Once()

		_, err := svc.Get(ctx, spaceID, userID)
		require.NoError(t, err)

		rl, err := svc.Get(ctx, spaceID, userID)
		require.NoError(t, err)
		assert.Equal(t, spaceRole, rl)

		cs.AssertExpectations(t)
	})

	t.Run("ListCollaborators from cache", func(t *testing.T) {
		cs := &csmocks.Collaborators{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

		cs.On("ListCollaborators", mock.Anything, spaceID).Return([]*collaborators.Collaborator{{SpaceID: spaceID, Subject: userID, Role: spaceRole}}, nil).Once()

		v1, err := svc.ListCollaborators(ctx, spaceID)
		require.NoError(t, err)
		v2, err := svc.ListCollaborators(ctx, spaceID)
		require.NoError(t, err)
		assert.Same(t, v1[0], v2[0], "Ожидается получение объектов из кэша при повторном запросе.")

		cs.AssertExpectations(t)
	})

	t.Run("ListSpaces from cache", func(t *testing.T) {
		cs := &csmocks.Collaborators{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

		cs.On("ListSpaces", mock.Anything, userID).Return([]*collaborators.Collaborator{{SpaceID: spaceID, Subject: userID, Role: spaceRole}}, nil).Once()

		v1, err := svc.ListSpaces(ctx, userID)
		require.NoError(t, err)
		v2, err := svc.ListSpaces(ctx, userID)
		require.NoError(t, err)
		assert.Same(t, v1[0], v2[0], "Ожидается получение объектов из кэша при повторном запросе.")

		cs.AssertExpectations(t)
	})

	t.Run("Invalidate cache", func(t *testing.T) {
		t.Run("After Remove", func(t *testing.T) {
			cs := &csmocks.Collaborators{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			cs.On("Get", mock.Anything, spaceID, userID).Return(spaceRole, nil).Once()
			cs.On("ListCollaborators", mock.Anything, spaceID).Return([]*collaborators.Collaborator{{SpaceID: spaceID, Subject: userID, Role: spaceRole}}, nil).Once()
			cs.On("ListSpaces", mock.Anything, userID).Return([]*collaborators.Collaborator{{SpaceID: spaceID, Subject: userID, Role: spaceRole}}, nil).Once()

			_, err := svc.Get(ctx, spaceID, userID)
			require.NoError(t, err)

			rl, err := svc.Get(ctx, spaceID, userID)
			require.NoError(t, err)
			assert.Equal(t, spaceRole, rl, "Ожидается получение данных из кэша.")

			lc1, err := svc.ListCollaborators(ctx, spaceID)
			require.NoError(t, err)
			lc2, err := svc.ListCollaborators(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, lc1[0], lc2[0], "Ожидается получение объектов из кэша.")

			ls1, err := svc.ListSpaces(ctx, userID)
			require.NoError(t, err)
			ls2, err := svc.ListSpaces(ctx, userID)
			require.NoError(t, err)
			assert.Same(t, ls1[0], ls2[0], "Ожидается получение объектов из кэша.")

			cs.On("Remove", mock.Anything, spaceID, userID).Return(nil).Once()

			cs.On("Get", mock.Anything, spaceID, userID).Return("", errNotFound).Once()
			cs.On("ListCollaborators", mock.Anything, spaceID).Return(nil, errNotFound).Once()
			cs.On("ListSpaces", mock.Anything, userID).Return(nil, errNotFound).Once()

			err = svc.Remove(ctx, spaceID, userID)

			rl, err = svc.Get(ctx, spaceID, userID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается удаление данных из кеша, и получение ошибки от сервиса")
			assert.Empty(t, rl)

			lc, err := svc.ListCollaborators(ctx, spaceID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается удаление данных из кеша, и получение ошибки от сервиса")
			assert.Nil(t, lc)

			ls, err := svc.ListSpaces(ctx, userID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается удаление данных из кеша, и получение ошибки от сервиса")
			assert.Nil(t, ls)

			cs.AssertExpectations(t)
		})

		t.Run("After TTL expired", func(t *testing.T) {
			cs := &csmocks.Collaborators{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			cs.On("Get", mock.Anything, spaceID, userID).Return(spaceRole, nil).Once()
			cs.On("ListCollaborators", mock.Anything, spaceID).Return([]*collaborators.Collaborator{{SpaceID: spaceID, Subject: userID, Role: spaceRole}}, nil).Once()
			cs.On("ListSpaces", mock.Anything, userID).Return([]*collaborators.Collaborator{{SpaceID: spaceID, Subject: userID, Role: spaceRole}}, nil).Once()

			_, err := svc.Get(ctx, spaceID, userID)
			require.NoError(t, err)

			rl, err := svc.Get(ctx, spaceID, userID)
			require.NoError(t, err)
			assert.Equal(t, spaceRole, rl, "Ожидается получение данных из кэша.")

			lc1, err := svc.ListCollaborators(ctx, spaceID)
			require.NoError(t, err)
			lc2, err := svc.ListCollaborators(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, lc1[0], lc2[0], "Ожидается получение объектов из кэша.")

			ls1, err := svc.ListSpaces(ctx, userID)
			require.NoError(t, err)
			ls2, err := svc.ListSpaces(ctx, userID)
			require.NoError(t, err)
			assert.Same(t, ls1[0], ls2[0], "Ожидается получение объектов из кэша.")

			cs.On("Remove", mock.Anything, spaceID, userID).Return(nil).Once()

			cs.On("Get", mock.Anything, spaceID, userID).Return("", errNotFound).Once()
			cs.On("ListCollaborators", mock.Anything, spaceID).Return(nil, errNotFound).Once()
			cs.On("ListSpaces", mock.Anything, userID).Return(nil, errNotFound).Once()

			err = svc.Remove(ctx, spaceID, userID)

			rl, err = svc.Get(ctx, spaceID, userID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается удаление данных из кеша, и получение ошибки от сервиса")
			assert.Empty(t, rl)

			lc, err := svc.ListCollaborators(ctx, spaceID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается удаление данных из кеша, и получение ошибки от сервиса")
			assert.Nil(t, lc)

			ls, err := svc.ListSpaces(ctx, userID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается удаление данных из кеша, и получение ошибки от сервиса")
			assert.Nil(t, ls)

			cs.AssertExpectations(t)
		})
	})

}
