package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/clients"
	csmocks "git.perx.ru/perxis/perxis-go/pkg/clients/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestClientsCache(t *testing.T) {

	const (
		cltID    = "cltID"
		spaceID  = "spaceID"
		clientID = "123@client"
		size     = 5
		ttl      = 20 * time.Millisecond
	)

	errNotFound := errors.NotFound(errors.New("not found"))

	ctx := context.Background()

	t.Run("Get from cache", func(t *testing.T) {
		cs := &csmocks.Clients{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

		cs.On("Get", mock.Anything, spaceID, cltID).Return(&clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}, nil).Once()

		v1, err := svc.Get(ctx, spaceID, cltID)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, spaceID, cltID)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается получение объекта из кэша, после повторного запроса.")

		v3, err := svc.GetBy(ctx, spaceID, &clients.GetByParams{OAuthClientID: clientID})
		require.NoError(t, err)
		assert.Same(t, v2, v3, "Ожидается получение объекта из кэша при запросе по ClientID.")

		cs.AssertExpectations(t)
	})

	t.Run("GetBy from cache", func(t *testing.T) {
		cs := &csmocks.Clients{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

		cs.On("GetBy", mock.Anything, spaceID, &clients.GetByParams{OAuthClientID: clientID}).Return(&clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}, nil).Once()

		v1, err := svc.GetBy(ctx, spaceID, &clients.GetByParams{OAuthClientID: clientID})
		require.NoError(t, err)

		v2, err := svc.GetBy(ctx, spaceID, &clients.GetByParams{OAuthClientID: clientID})
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается получение объекта из кэша, после повторного запроса.")

		v3, err := svc.Get(ctx, spaceID, cltID)
		require.NoError(t, err)
		assert.Same(t, v2, v3, "Ожидается получение объекта из кэша, после запроса Get.")

		cs.AssertExpectations(t)
	})

	t.Run("List", func(t *testing.T) {
		cs := &csmocks.Clients{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

		cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}}, nil).Once()

		vl1, err := svc.List(ctx, spaceID)
		require.NoError(t, err)

		vl2, err := svc.List(ctx, spaceID)
		require.NoError(t, err)
		assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша, после повторного запроса.")

		cs.AssertExpectations(t)
	})

	t.Run("Invalidate cache", func(t *testing.T) {

		t.Run("After Update", func(t *testing.T) {
			cs := &csmocks.Clients{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			cs.On("Get", mock.Anything, spaceID, cltID).Return(&clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}, nil).Once()
			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			v3, err := svc.GetBy(ctx, spaceID, &clients.GetByParams{OAuthClientID: clientID})
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кэша по ClientID.")

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Len(t, vl2, 1)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша, после повторного запроса.")

			cs.On("Update", mock.Anything, mock.Anything).Return(nil).Once()

			err = svc.Update(ctx, &clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_2", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}})
			require.NoError(t, err)

			cs.On("Get", mock.Anything, spaceID, cltID).Return(&clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_2", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}, nil).Once()
			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, Name: "client_2", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}}, nil).Once()

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидается получение объектов из кэша, после повторного запроса.")

			v4, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v4, "Ожидает что после обновления объект был удален из кэша и будет запрошен заново из сервиса.")

			v5, err := svc.GetBy(ctx, spaceID, &clients.GetByParams{OAuthClientID: clientID})
			require.NoError(t, err)
			assert.NotSame(t, v3, v5)
			assert.Same(t, v4, v5, "Ожидается что после обновления объект был удален из кеша и после запроса Get в кеш попал объект запрошенный заново из сервиса.")

			cs.AssertExpectations(t)
		})

		t.Run("After Update(List)", func(t *testing.T) {
			cs := &csmocks.Clients{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}}, nil).Once()

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Len(t, vl2, 1)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша, после повторного запроса.")

			cs.On("Update", mock.Anything, mock.Anything).Return(nil).Once()

			err = svc.Update(ctx, &clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_2", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}})
			require.NoError(t, err)

			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, Name: "client_2", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}}, nil).Once()

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидается получение объектов из кэша, после повторного запроса.")

			cs.AssertExpectations(t)
		})

		t.Run("After Delete", func(t *testing.T) {
			cs := &csmocks.Clients{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			cs.On("Get", mock.Anything, spaceID, cltID).Return(&clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}, nil).Once()
			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			v3, err := svc.GetBy(ctx, spaceID, &clients.GetByParams{OAuthClientID: clientID})
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кэша по ClientID.")

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша, после повторного запроса.")

			cs.On("Delete", mock.Anything, spaceID, cltID).Return(nil).Once()

			err = svc.Delete(ctx, spaceID, cltID)
			require.NoError(t, err)

			cs.On("Get", mock.Anything, spaceID, cltID).Return(nil, errNotFound).Once()
			cs.On("GetBy", mock.Anything, spaceID, &clients.GetByParams{OAuthClientID: clientID}).Return(nil, errNotFound).Once()
			cs.On("List", mock.Anything, spaceID).Return(nil, errNotFound).Once()

			_, err = svc.Get(ctx, spaceID, cltID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается что после удаление из хранилища объект был удален из кэша и получена ошибка из сервиса.")

			_, err = svc.GetBy(ctx, spaceID, &clients.GetByParams{OAuthClientID: clientID})
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается что после удаление из хранилища объект был удален из кэша и получена ошибка из сервиса.")

			_, err = svc.List(ctx, spaceID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается что после удаление из хранилища объекты были удалены из кэша.")

			cs.AssertExpectations(t)
		})

		t.Run("After Delete(List)", func(t *testing.T) {
			cs := &csmocks.Clients{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}}}, nil).Once()

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша, после повторного запроса.")

			cs.On("Delete", mock.Anything, spaceID, cltID).Return(nil).Once()

			err = svc.Delete(ctx, spaceID, cltID)
			require.NoError(t, err)

			cs.On("List", mock.Anything, spaceID).Return(nil, errNotFound).Once()

			_, err = svc.List(ctx, spaceID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается что после удаление из хранилища объекты были удалены из кэша.")

			cs.AssertExpectations(t)
		})

		t.Run("After Create", func(t *testing.T) {
			cs := &csmocks.Clients{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, Name: "client_1"}}, nil).Once()

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша, после повторного запроса.")
			assert.Len(t, vl2, 1, "Ожидается получение объектов из кэша.")

			cs.On("Create", mock.Anything, mock.Anything).Return(&clients.Client{ID: "cltID2", SpaceID: spaceID, Name: "client_2"}, nil).Once()

			_, err = svc.Create(ctx, &clients.Client{ID: "cltID2", SpaceID: spaceID, Name: "client_2"})
			require.NoError(t, err)

			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, Name: "client_1"}, {ID: "cltID2", SpaceID: spaceID, Name: "client_2"}}, nil).Once()

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Len(t, vl3, 2, "Ожидается что после создания нового объекта кеш будет очищен и объекты запрошены заново из сервиса.")

			cs.AssertExpectations(t)
		})

		t.Run("After Enable", func(t *testing.T) {
			cs := &csmocks.Clients{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			tr := true
			cs.On("Get", mock.Anything, spaceID, cltID).Return(&clients.Client{ID: cltID, SpaceID: spaceID, OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}, Disabled: &tr}, nil).Once()
			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}, Disabled: &tr}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			v3, err := svc.GetBy(ctx, spaceID, &clients.GetByParams{OAuthClientID: clientID})
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кэша по ClientID.")

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша, после повторного запроса.")

			cs.On("Enable", mock.Anything, spaceID, cltID, tr).Return(nil).Once()

			err = svc.Enable(ctx, spaceID, cltID, tr)
			require.NoError(t, err)

			fl := false
			cs.On("Get", mock.Anything, spaceID, cltID).Return(&clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}, Disabled: &fl}, nil).Once()
			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}, Disabled: &fl}}, nil).Once()

			v4, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v4, "Ожидается что после активации объект был удален из кэша и запрошен у сервиса.")

			v5, err := svc.GetBy(ctx, spaceID, &clients.GetByParams{OAuthClientID: clientID})
			assert.NotSame(t, v3, v5, "Ожидается что после активации объект был удален из кеша и после запроса Get в кеш попал объект запрошенный заново из сервиса.")

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидается что после активации объекта, кеш будет очищен и объекты будут запрошены заново из сервиса.")

			cs.AssertExpectations(t)
		})

		t.Run("After Enable(List)", func(t *testing.T) {
			cs := &csmocks.Clients{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			tr := true
			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}, Disabled: &tr}}, nil).Once()

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша, после повторного запроса.")

			cs.On("Enable", mock.Anything, spaceID, cltID, tr).Return(nil).Once()

			err = svc.Enable(ctx, spaceID, cltID, tr)
			require.NoError(t, err)

			fl := false
			cs.On("List", mock.Anything, spaceID).Return([]*clients.Client{{ID: cltID, SpaceID: spaceID, OAuth: &clients.OAuth{ClientID: clientID, AuthID: "authID"}, Disabled: &fl}}, nil).Once()

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидается что после активации объекта, кеш будет очищен и объекты будут запрошены заново из сервиса.")

			cs.AssertExpectations(t)
		})

		t.Run("After TTL expired", func(t *testing.T) {
			cs := &csmocks.Clients{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(cs)

			cs.On("Get", mock.Anything, spaceID, cltID).Return(&clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID}}, nil).Once()
			cs.On("Get", mock.Anything, spaceID, cltID).Return(&clients.Client{ID: cltID, SpaceID: spaceID, Name: "client_1", OAuth: &clients.OAuth{ClientID: clientID}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)
			v2, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша после повторного запроса.")

			time.Sleep(2 * ttl)

			v3, err := svc.Get(ctx, spaceID, cltID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v3, "Ожидается что элемент был удален из кэша по истечению ttl и будет запрошен заново из сервиса.")

			cs.AssertExpectations(t)
		})
	})
}
