package clients

// Client - приложение имеющее доступ к API
type Client struct {
	// Внутренний идентификатор клиента внутри системы
	ID string `json:"id" bson:"_id"`

	// Идентификатор пространства
	SpaceID string `json:"space_id" bson:"-"`

	// Имя приложения (обязательное поле)
	Name string `json:"name" bson:"name"`

	// Параметры аутентификации клиента
	OAuth  *OAuth  `json:"oauth,omitempty" bson:"oauth,omitempty"`
	TLS    *TLS    `json:"tls,omitempty" bson:"tls,omitempty"`
	APIKey *APIKey `json:"api_key,omitempty" bson:"api_key,omitempty"`

	// Описание клиента, назначение
	Description string `json:"description" bson:"description"`

	// Приложение отключено и не может авторизоваться
	Disabled *bool `json:"disabled,omitempty" bson:"disabled,omitempty"`

	// Роль приложения в пространстве
	RoleID string `json:"role_id" bson:"role_id"`
}

type OAuth struct {
	ClientID     string `bson:"client_id,omitempty" json:"client_id,omitempty"`         // Идентификатор клиента выданные IdP сервером, используется для идентификации клиента
	AuthID       string `bson:"auth_id,omitempty" json:"auth_id,omitempty"`             // Сервис, который используется для авторизации клиента
	TokenURL     string `bson:"token_url,omitempty" json:"token_url,omitempty"`         // URL для получения/обновления access token клиента (опционально)
	ClientSecret string `bson:"client_secret,omitempty" json:"client_secret,omitempty"` // Секретный Ключ клиента, используется для идентификации клиента (опционально)
}

type APIKey struct {
	Key    string `bson:"key,omitempty" json:"key,omitempty"`
	Rotate bool   `bson:"-" json:"rotate,omitempty"`
}

type TLS struct {
	Subject string `json:"subject,omitempty"`
	CACert  string `json:"ca_cert,omitempty"`
	Cert    string `json:"cert,omitempty"`
	Key     string `json:"key,omitempty"`
}

func (c *Client) SetDisabled(b bool) *Client {
	c.Disabled = &b
	return c
}

func (c *Client) IsDisabled() bool {
	if c.Disabled != nil && *c.Disabled {
		return true
	}
	return false
}

func (c Client) Clone() *Client {
	clone := &Client{
		ID:          c.ID,
		SpaceID:     c.SpaceID,
		Name:        c.Name,
		Description: c.Description,
		RoleID:      c.RoleID,
	}

	if c.OAuth != nil {
		temp := *c.OAuth
		clone.OAuth = &temp
	}
	if c.TLS != nil {
		temp := *c.TLS
		clone.TLS = &temp
	}
	if c.APIKey != nil {
		temp := *c.APIKey
		clone.APIKey = &temp
	}
	if c.Disabled != nil {
		temp := *c.Disabled
		clone.Disabled = &temp
	}

	return clone
}
