package cache

import (
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestCache(t *testing.T) {
	t.Run("Simple", func(t *testing.T) {
		c := NewCache(10, 0)

		{
			val, err := c.Get("test_key")
			require.Error(t, err)
			assert.EqualError(t, err, "not found")
			assert.Nil(t, val)
		}
		{
			err := c.Set("test_key", "test_val")
			require.NoError(t, err)

			val, err := c.Get("test_key")
			require.NoError(t, err)
			assert.Equal(t, "test_val", val.(string))
		}

		{
			err := c.Remove("test_key")
			require.NoError(t, err)

			val, err := c.Get("test_key")
			assert.EqualError(t, err, "not found")
			assert.Nil(t, val)
		}
	})
	t.Run("Value Evicted", func(t *testing.T) {
		c := NewCache(1, 0)

		{
			err := c.Set("test_key_1", "test_val_1")
			require.NoError(t, err)

			val, err := c.Get("test_key_1")
			require.NoError(t, err)
			assert.Equal(t, "test_val_1", val.(string))
		}

		{
			err := c.Set("test_key_2", "test_val_2")
			require.NoError(t, err)

			val, err := c.Get("test_key_1")
			assert.EqualError(t, err, "not found")
			assert.Nil(t, val)
			val, err = c.Get("test_key_2")
			require.NoError(t, err)
			assert.Equal(t, "test_val_2", val.(string))
		}

	})
	t.Run("TTL expired", func(t *testing.T) {
		c := NewCache(10, 10*time.Millisecond)

		err := c.Set("test_key", "test_val")
		require.NoError(t, err)

		val, err := c.Get("test_key")
		require.NoError(t, err)
		assert.Equal(t, "test_val", val.(string))

		time.Sleep(15 * time.Millisecond)

		val, err = c.Get("test_key")
		assert.EqualError(t, err, "not found")
		assert.Nil(t, val)
	})
}
