package actionurl

import (
	"fmt"
	"net/url"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestActionURL_New(t *testing.T) {
	tests := []struct {
		name    string
		action  string
		want    *ActionURL
		url     string
		wantErr assert.ErrorAssertionFunc
	}{
		{
			name:    "Without action",
			want:    &ActionURL{},
			wantErr: assert.NoError,
		},
		{
			name:    "Without deprecated action call",
			action:  "build-site",
			want:    &ActionURL{},
			url:     "build-site",
			wantErr: assert.NoError,
		},
		{
			name:   "With grpc action",
			action: "grpc:///perxisweb/build-site",
			want: &ActionURL{
				id:        "build-site",
				extension: "perxisweb",
			},
			url:     "grpc:///perxisweb/build-site",
			wantErr: assert.NoError,
		},
		{
			name:   "With ui action",
			action: "ui:///space/env/coll",
			want: &ActionURL{
				id:        "",
				extension: "",
			},
			url:     "ui:///space/env/coll",
			wantErr: assert.NoError,
		},
		{
			name:    "With http action",
			action:  "https://perx.ru",
			want:    &ActionURL{},
			url:     "https://perx.ru",
			wantErr: assert.NoError,
		},
		{
			name:    "With error in parse",
			action:  "grpc://user:abc{DEf1=ghi@example.com:5432/db?sslmode=require",
			want:    nil,
			wantErr: assert.Error,
		},
		{
			name:    "With no action id",
			action:  "grpc:///perxisweb",
			want:    nil,
			wantErr: assert.Error,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if tt.url != "" {
				tt.want.URL, _ = url.Parse(tt.url)
			}
			got, err := New(tt.action)
			if !tt.wantErr(t, err, fmt.Sprintf("New(%v)", tt.action)) {
				return
			}
			assert.Equalf(t, tt.want, got, "New(%v)", tt.action)
		})
	}
}

func TestActionURL_Make(t *testing.T) {
	tests := []struct {
		name      string
		id        string
		extension string
		url       string
		want      string
	}{
		{
			name: "Without action and extensions id's #1",
			url:  "grpc:///extension-id/action-id",
			want: "grpc:///extension-id/action-id",
		},
		{
			name: "Without action and extensions id's #2",
			url:  "ui:///space/env/coll",
			want: "ui:///space/env/coll",
		},
		{
			name: "Without action and extensions id's #3",
			url:  "space/env/coll",
			want: "space/env/coll",
		},
		{
			name: "Without action and extensions id's #4",
			url:  "https://perx.ru",
			want: "https://perx.ru",
		},
		{
			name:      "With action and extensions",
			id:        "action-id",
			extension: "extension-id",
			want:      "grpc:///extension-id/action-id",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &ActionURL{
				id:        tt.id,
				extension: tt.extension,
			}
			if tt.url != "" {
				p.URL, _ = url.Parse(tt.url)
			}
			assert.Equalf(t, tt.want, p.Make(), "MakeFromURL()")
		})
	}
}
