package actionurl

import (
	"fmt"
	"net/url"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
)

// ActionURL структура для хранения данных о переданном действии.
type ActionURL struct {
	id        string
	extension string
	*url.URL
}

// New возвращает структуру ActionURL
func New(action string) (*ActionURL, error) {
	actionURL := &ActionURL{}

	if action == "" {
		return actionURL, nil
	}

	err := actionURL.SetURL(action)
	if err != nil {
		return nil, err
	}
	if actionURL.URL.Scheme == "grpc" {
		path := actionURL.Path
		if strings.HasPrefix(actionURL.Path, "/") {
			path = actionURL.Path[1:]
		}
		splitPath := strings.Split(path, "/")
		if len(splitPath) < 2 {
			return nil, errors.Errorf("incorrect action URL, no action id: '%s'", action)
		}
		actionURL.extension = splitPath[0]
		actionURL.id = splitPath[1]
	}

	return actionURL, nil
}

// ID возвращает сохраненный в ActionURL id действия
func (p *ActionURL) ID() string {
	return p.id
}

// SetID устанавливает в ActionURL id действия
func (p *ActionURL) SetID(id string) {
	p.id = id
}

// Extension возвращает сохраненный в ActionURL id расширения
func (p *ActionURL) Extension() string {
	return p.extension
}

// SetExtension устанавливает в ActionURL id расширения
func (p *ActionURL) SetExtension(ext string) {
	p.extension = ext
}

// SetURL устанавливает структуру URL
func (p *ActionURL) SetURL(u string) (err error) {
	if p.URL, err = url.Parse(u); err != nil {
		return err
	}
	return nil
}

// Make возвращает Action URL из сохраненных данных
func (p *ActionURL) Make() string {
	if p.id != "" && p.extension != "" {
		return fmt.Sprintf("grpc:///%s/%s", p.extension, p.id)
	}
	return p.URL.String()
}
